<?php

namespace Inside\Console\Commands;

use Composer\Json\JsonFile;
use Exception;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;
use Inside\Facades\Package;
use Inside\Support\InsidePackage;
use Symfony\Component\Process\Exception\ProcessFailedException;
use Symfony\Component\Process\Process;

class UpdateCodeCommand extends InstallationTools
{
    protected $signature = 'inside:update-code
                            {branch?}
                            {--force : Mettre à jour le code sans demander de confirmation}
                            {--no-reset : Ne pas reset hard avant de pull}
                            {--no-checkout : Ne pas checkout avant de pull}
                            {--packages= : Mettre à jour des packages en particulier}
                            {--no-cleaning : Ne pas lancer le nettoyage d\'Inside}';

    protected $description = 'Mettre à jour le code d\'inside';

    /** @var mixed|null */
    protected $composerLock = null;

    public function handle(): void
    {
        /** @var string|null $branch */
        $branch = $this->argument('branch');
        $this->line("Mise-à-jour du code d'inside");
        if ($branch) {
            $this->info("Basculement des branches sur [$branch]");
        } else {
            $this->info('Respect des branches en cours');
        }
        $jsonPath = cms_base_path('composer.lock');
        if (! File::exists($jsonPath)) {
            $this->dumpAutoloads();
        }
        $jsonFile = null;
        if (File::exists($jsonPath)) {
            $jsonFile = new JsonFile($jsonPath);
            try {
                $this->composerLock = $jsonFile->read();
            } catch (Exception $e) {
                $this->error($e->getMessage());
            }
        }
        if (
            ! $this->option('force')
            && ! $this->confirm(
                'Vous êtes sur le point de mettre le code d\'inside à jour, êtes vous sur de vouloir continuer ?'
            )
        ) {
            return;
        }
        $this->call('inside:down');

        if ($this->option('no-checkout')) {
            $this->writeln('<fg=cyan>'.str_repeat('-', 80).'</fg=cyan>');
            $this->comment('Le checkout est désactivé: le code sera chargé dans la branche où il se trouve!');
            $this->writeln('<fg=cyan>'.str_repeat('-', 80).'</fg=cyan>');
        }

        if ($this->option('no-reset')) {
            $this->writeln('<fg=magenta>'.str_repeat('-', 80).'</fg=magenta>');
            $this->comment('Le reset est désactivé: le code sera chargé sans reset --hard au préalable');
            $this->writeln('<fg=magenta>'.str_repeat('-', 80).'</fg=magenta>');
        }

        $this->call('cache:clear');
        /** @var Collection<InsidePackage>|InsidePackage[] $modules */
        $modules = Package::list();

        /** @var string|null $packages */
        $packages = $this->option('packages');
        $packages = ! empty($packages) ? explode(',', $packages) : null;

        foreach ($modules as $module) {
            if (! empty($packages) && ! in_array($module->getName(), $packages)) {
                continue;
            }

            $this->updateModule($module, $branch ?? '');
        }

        if ($jsonFile instanceof JsonFile) {
            try {
                $jsonFile->write($this->composerLock);
            } catch (Exception $e) {
                $this->error($e->getMessage());
            }
        }
        if ($this->option('migrate')) {
            $this->drush('cr');
            $this->call('migrate');
        }
        $this->call('inside:update', ['--no-cleaning' => $this->option('no-cleaning')]);
        $this->comment('[[Inside prêt]]');
    }

    protected function updateModule(InsidePackage $module, string $branch = ''): void
    {
        $this->info("Mise-à-jour du module <fg=cyan>[{$module->getName()}]</fg=cyan>");
        $this->writeln(
            "<info>Version détectée dans composer.lock <fg=blue>[{$module->getVersion()}]</fg=blue> <fg=yellow>({$module->getCommit()})</fg=yellow></info>"
        );

        if ($branch == '') {
            $branch = $module->getVersion();
        }

        if (Str::startsWith($branch, 'dev-')) {
            $branch = substr($branch, 4);
        }
        $this->write('Récupération du code');

        $modulePath = cms_base_path($module->getPath().$module->getName());
        $shouldReset = ! $this->option('no-reset');
        $shouldCheckout = ! $this->option('no-checkout');

        $cmd = "cd $modulePath"
            .($shouldReset ? ' && git reset --hard' : '')
            .($shouldCheckout ? " && git checkout $branch" : '')
            .' && git fetch && git pull';

        $process = Process::fromShellCommandLine($cmd, null, null, null, 300);

        try {
            $process->run();
        } catch (Exception $exception) {
            throw new ProcessFailedException($process);
        }
        if (! $process->isSuccessful()) {
            $this->writeResult(false);

            return;
        }
        $cmd = 'cd '.cms_base_path($module->getPath().$module->getName())
            .'&& git log -1 --pretty=format:\'{"commit":"%H"}\'';
        $process = Process::fromShellCommandLine($cmd);
        $process->run();
        $logResult = json_decode(trim($process->getOutput()), true);

        if (! is_null($logResult)) {
            $this->writeResult(true);
            $this->info(
                "<fg=black>[{$module->getVersion()}]</fg=black> mis-à-jour en <fg=yellow>{$logResult['commit']}</fg=yellow>"
            );
            $this->separatorLine();
        }
    }
}
