<?php

namespace Inside\Console\Commands;

use Illuminate\Contracts\Console\Kernel as ConsoleKernelContract;
use Illuminate\Contracts\Filesystem\FileNotFoundException;
use Illuminate\Filesystem\Filesystem;
use Illuminate\Routing\RouteCollection;
use Illuminate\Support\Facades\Route;
use Inside\Application;
use Inside\Console\Command;
use Inside\Support\SerializableClosure\SerializableClosure;
use Opis\Closure\SerializableClosure as OpisSerializableClosure;
use Symfony\Component\Routing\Route as SymfonyRoute;

class RouteCacheCommand extends Command
{
    protected $name = 'inside:route:cache';

    protected $description = 'Mise en cache des routes enregistrées';

    /** @var Application */
    protected $laravel;

    public function __construct(
        protected Filesystem $files
    ) {
        parent::__construct();
    }

    /**
     * @throws FileNotFoundException
     */
    public function handle(): void
    {
        $this->call('inside:route:clear');

        $routes = $this->getFreshApplicationRoutes();

        if (count($routes) === 0) {
            $this->error('Inside n\'a enregistré aucune route !!');

            return;
        }

        $routeCollections = [];
        foreach ($routes as $routeKey => $route) {
            $routeCollections[] = $this->prepareForSerialization($routeKey, $route);
        }

        $this->files->put(
            $this->laravel->getCachedRoutesPath(),
            $this->buildRouteCacheFile($routeCollections)
        );

        $this->info('Routes cached successfully!');
    }

    protected function getFreshApplicationRoutes(): array
    {
        return Route::getRoutes();
    }

    /**
     * Get a fresh application instance.
     *
     * @return \Illuminate\Contracts\Foundation\Application
     */
    protected function getFreshApplication(): \Illuminate\Contracts\Foundation\Application
    {
        return tap(require $this->laravel->bootstrapPath().'/app.php', function ($app) {
            $app->make(ConsoleKernelContract::class)->bootstrap();
        });
    }

    /**
     * @throws FileNotFoundException
     */
    protected function buildRouteCacheFile(RouteCollection|array $routeCacheFile): string
    {
        $stub = $this->files->get(__DIR__.'/stubs/routes.stub');

        /** @var string $routeCacheFile */
        $routeCacheFile = str_replace('{{routes}}', base64_encode(serialize($routeCacheFile)), $stub);

        return $routeCacheFile;
    }

    public function prepareForSerialization(string $routeKey, array $route): object
    {
        $route = (object) $route;

        $route->key = $routeKey;

        if (! isset($route->action['uses'])) {
            $route->action['uses'] = $route->action[0];
            unset($route->action[0]);
        }
        if ($route->action['uses'] instanceof \Closure) {
            $route->action['uses'] = serialize(
                \PHP_VERSION_ID < 70400
                    ? new OpisSerializableClosure($route->action['uses'])
                    : new SerializableClosure($route->action['uses'])
            );
        }

        if (isset($route->action['missing']) && $route->action['missing'] instanceof \Closure) {
            $route->action['missing'] = serialize(
                \PHP_VERSION_ID < 70400
                    ? new OpisSerializableClosure($route->action['missing'])
                    : new SerializableClosure($route->action['missing'])
            );
        }

        return $route; //$this->toSymfonyRoute($route)->compile();
    }

    /**
     * @param mixed $route
     * @return SymfonyRoute
     */
    public function toSymfonyRoute($route): SymfonyRoute
    {
        return new SymfonyRoute(
            preg_replace('/\{(\w+?)\?\}/', '{$1}', $route->uri),
            $this->getOptionalParameterNames($route),
            [],
            ['utf8' => true, 'action' => $route->action],
            '',
            [],
            [$route->method]
        );
    }

    /**
     * @param mixed $route
     * @return array
     */
    protected function getOptionalParameterNames($route): array
    {
        preg_match_all('/\{(\w+?)\?\}/', $route->uri, $matches);

        return isset($matches[1]) ? array_fill_keys($matches[1], null) : [];
    }
}
