<?php

namespace Inside\Console\Commands;

use Illuminate\Contracts\Filesystem\FileNotFoundException;
use Illuminate\Filesystem\Filesystem;
use Illuminate\Support\Composer;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;
use Inside\Console\Command;
use Inside\Facades\Package;
use InvalidArgumentException;

class MigrateMakeCommand extends Command
{
    protected $signature = 'inside:make:migration {name : Le nom de la migration} {types* : Le ou les noms inside du ou des nouveaux types} {--module=}';

    public function __construct(
        protected Composer $composer,
        protected Filesystem $files
    ) {
        parent::__construct();
    }

    /**
     * @throws FileNotFoundException
     */
    public function handle(): void
    {
        /** @var string $name */
        $name = $this->input->getArgument('name');
        $name = Str::snake(trim($name));

        /** @var array $types */
        $types = $this->input->getArgument('types');

        /** @var string $module */
        $module = $this->input->getOption('module') ?? '';

        $this->writeMigration($name, $types, $module);

        $this->composer->dumpAutoloads();
    }

    /**
     * @throws FileNotFoundException
     */
    protected function writeMigration(string $name, array $types, string $module = ''): void
    {
        $this->makeMigration($name, $types, $module);

        foreach ($types as $type) {
            $this->makeBridgeMigration($type);
        }

        $this->line("<info>Migration 'bridge' créée:</info> {$name}");
    }

    protected function ensureMigrationDoesntAlreadyExist(string $name, string $migrationPath): void
    {
        if (! empty($migrationPath)) {
            $migrationFiles = $this->files->glob($migrationPath.'/*.php');

            foreach ($migrationFiles as $migrationFile) {
                $this->files->requireOnce($migrationFile);
            }
        }

        if (class_exists($className = $this->getClassName($name))) {
            throw new InvalidArgumentException("Une classe {$className} existe déjà, choisissez un autre nom de migration.");
        }
    }

    /**
     * @throws FileNotFoundException
     */
    protected function makeMigration(string $name, array $types, string $module = ''): void
    {
        $path = $this->getMigrationPath($module);
        if (! File::exists($path)) {
            File::makeDirectory($path);
        }
        $this->ensureMigrationDoesntAlreadyExist($name, $path);

        $stub = $this->getStub();
        $bridgeStub = $this->getBridgeStub();

        $this->files->put(
            $this->getPath($name, $path),
            $this->populateStub($name, $types, $stub, $bridgeStub)
        );
    }

    protected function makeBridgeMigration(string $type): void
    {
        $stub = $this->getTypeStub();

        $path = $this->getBridgeMigrationPath($type);
        if (! File::exists($path)) {
            File::makeDirectory($path, 0755, true);
        }
        $this->files->put(
            $this->getBridgePath($path),
            $this->populateBridgeStub($type, $stub)
        );
    }

    protected function populateStub(string $name, array $types, string $stub, string $bridgeStub): string
    {
        $stub = str_replace('DummyClass', $this->getClassName($name), $stub);

        $bridgeMigration = '';
        foreach ($types as $type) {
            $bridgeMigration .= str_replace('dummy_type', $type, $bridgeStub);
        }

        return str_replace('@@INSIDE_BRIDGE_MIGRATION@@', $bridgeMigration, $stub);
    }

    protected function populateBridgeStub(string $type, string $stub): string
    {
        $stub = str_replace('dummy_type', $type, $stub);

        return str_replace('dummy_label', Str::studly($type), $stub);
    }

    /**
     * @throws FileNotFoundException
     */
    protected function getStub(): string
    {
        return $this->files->get($this->stubPath().'/create.stub');
    }

    /**
     * @throws FileNotFoundException
     */
    protected function getBridgeStub(): string
    {
        return $this->files->get($this->stubPath().'/bridge_create.stub');
    }

    /**
     * @throws FileNotFoundException
     */
    protected function getTypeStub(): string
    {
        return $this->files->get($this->stubPath().'/bridge_type.stub');
    }

    protected function getMigrationPath(string $module = ''): string
    {
        if ($module === '') {
            return back_path('database/migrations');
        } else {
            if (! Package::has($module)) {
                throw new InvalidArgumentException(
                    "Le module {$module} n'est pas un module maecia valide ou n'est pas installé sur cette instance."
                );
            }

            return cms_base_path('vendor/maecia/'.$module.'/database/migrations');
        }
    }

    protected function getBridgeMigrationPath(string $type, string $module = ''): string
    {
        if ($module === '') {
            return back_path('database/contents/'.$type);
        } else {
            return cms_base_path('vendor/maecia/'.$module.'database/contents/'.$type);
        }
    }

    protected function getClassName(string $name): string
    {
        return Str::studly($name);
    }

    protected function stubPath(): string
    {
        return __DIR__.'/stubs';
    }

    protected function getDatePrefix(): string
    {
        static $date = null;
        if ($date == null) {
            return date('Y_m_d_His');
        }

        return $date;
    }

    protected function getPath(string $name, string $path): string
    {
        return $path.'/'.$this->getDatePrefix().'_'.$name.'.php';
    }

    protected function getBridgePath(string $path): string
    {
        return $path.'/'.$this->getDatePrefix().'.php';
    }
}
