<?php

namespace Inside\Console\Commands;

use Illuminate\Console\ConfirmableTrait;
use Illuminate\Encryption\Encrypter;
use Inside\Application;
use Inside\Console\Command;

class KeyGenerateCommand extends Command
{
    use ConfirmableTrait;

    /** @var Application */
    protected $laravel;

    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'key:generate
                    {--show : Display the key instead of modifying files}
                    {--force : Force the operation to run when in production}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Set the application key';

    /**
     * Execute the console command.
     */
    public function handle(): void
    {
        $key = $this->generateRandomKey();
        if ($this->option('show')) {
            $this->line('<comment>'.$key.'</comment>');

            return;
        }

        if (! $this->setKeyInEnvironmentFile($key)) {
            return;
        }
        $this->laravel['config']['app.key'] = $key;
        $this->info("Application key [$key] set successfully.");
    }

    /**
     * Generate a random key for the application.
     */
    protected function generateRandomKey(): string
    {
        return 'base64:'.base64_encode(
            Encrypter::generateKey($this->laravel['config']['app.cipher'])
        );
    }

    /**
     * Set the application key in the environment file.
     */
    protected function setKeyInEnvironmentFile(string $key): bool
    {
        $currentKey = $this->laravel['config']['app.key'];
        if (strlen($currentKey) !== 0 && (! $this->confirmToProceed())) {
            return false;
        }
        $this->writeNewEnvironmentFileWith($key);

        return true;
    }

    /**
     * Write a new environment file with the given key.
     */
    protected function writeNewEnvironmentFileWith(string $key): void
    {
        /** @var string $fileContents */
        $fileContents = file_get_contents($this->laravel->environmentFilePath());
        file_put_contents(
            $this->laravel->environmentFilePath(),
            preg_replace(
                $this->keyReplacementPattern(),
                'APP_KEY="'.$key.'"',
                $fileContents
            )
        );
    }

    /**
     * Get a regex pattern that will match env APP_KEY with any random key.
     */
    protected function keyReplacementPattern(): string
    {
        $escaped = '\="?'.preg_quote($this->laravel['config']['app.key'], '/').'"?';

        return "/^APP_KEY{$escaped}/m";
    }
}
