<?php

namespace Inside\Console\Commands;

use Drupal;
use Exception;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Inside\Authentication\Models\User;
use Inside\Console\Command;
use Inside\Content\Facades\Schema as InsideSchema;
use Inside\Host\Services\DrupalService;

/**
 * Class CleanupCommand
 */
class CleanupCommand extends Command
{
    protected $name = 'inside:cleanup';

    protected $signature = 'inside:cleanup {--dry-run}';

    protected $description = 'Supprime les données fantômes';

    public function line($string, $style = null, $verbosity = null): void
    {
        Log::info('[CleanupCommand] '.$string);
        parent::line($string, $style, $verbosity);
    }

    public function info($string, $verbosity = null): void
    {
        Log::info('[CleanupCommand] '.$string);
        parent::info($string, $verbosity);
    }

    public function comment($string, $verbosity = null): void
    {
        Log::info('[CleanupCommand] '.$string);
        parent::comment($string, $verbosity);
    }

    public function handle(): void
    {
        Artisan::call('cache:clear');

        $dryRun = $this->option('dry-run') === true;

        // Check drupal is started
        new DrupalService();

        $this->cleanUpGhostedContents($dryRun);

        $this->cleanUpAuthenticationUsers($dryRun);

        $this->cleanPivots($dryRun);
        $this->comment('Nettoyage terminé');
    }

    protected function cleanUpGhostedContents(bool $dryRun = false): void
    {
        $this->line('Nettoyage des contenus fantômes ( présent dans nos contenus mais non présent dans drupal )');
        foreach (InsideSchema::getTypes() as $type) {
            $this->line("Nettoyage des contenus de type [{$type}]");
            try {
                call_user_func(
                    /** @phpstan-ignore-next-line */
                    (InsideSchema::isContentType($type) ? type_to_class($type) : section_type_to_class($type))
                    .'::each',
                    function ($content) use ($type, $dryRun) {
                        $entityType = guess_drupal_entity_type(get_class($content)); // @phpstan-ignore-line
                        $entity = Drupal::service('entity.repository')->loadEntityByUuid($entityType, $content->uuid_host);
                        if (! $entity) {
                            $this->info(
                                "Le contenu de type {$type} pour uuid {$content->uuid} est fantôme et va être supprimé"
                            );
                            if (! $dryRun) {
                                $content->delete();
                            }
                        }
                    }
                );
            } catch (Exception $e) {
                $this->line("[{$type}] semble ne plus exister ! [".$e->getMessage().']');
            }
        }
    }

    protected function cleanUpAuthenticationUsers(bool $dryRun = false): void
    {
        $this->line("Nettoyage des comptes d'authentification fantômes");

        User::each(
            function ($user) use ($dryRun) {
                if (! $user->information) {
                    $this->info(
                        "Le compte {$user->email} / {$user->uuid} est fantôme et va être supprimé"
                    );

                    if (! $dryRun) {
                        $user->delete();
                    }
                }
            }
        );
    }

    private function deletePivot(int $id): void
    {
        DB::table('inside_pivots')->where('id', $id)->delete();
    }

    protected function cleanPivots(bool $dryRun = false): void
    {
        $this->line('Nettoyage des pivots fantômes');

        DB::table('inside_pivots')->get()->each(
            function ($pivot) use ($dryRun) {
                $parentType = class_to_type($pivot->parent_type);
                if (! InsideSchema::hasModel($parentType)) {
                    $this->info(
                        "Le pivot {$pivot->id} a un parent dont le type {$parentType} n'existe plus et va être supprimé"
                    );
                    if (! $dryRun) {
                        $this->deletePivot($pivot->id);
                    }

                    return;
                }
                $relatedType = class_to_type($pivot->related_type);
                if (! InsideSchema::hasModel($relatedType)) {
                    $this->info(
                        "Le pivot {$pivot->id} a une relation dont le type {$relatedType} n'existe plus et va être supprimé"
                    );
                    if (! $dryRun) {
                        $this->deletePivot($pivot->id);
                    }

                    return;
                }
                /** @phpstan-ignore-next-line */
                $parent = call_user_func($pivot->parent_type.'::find', $pivot->parent_uuid);
                if (! $parent) {
                    $this->info(
                        "Le pivot {$pivot->id} a perdu son parent et va être supprimé"
                    );
                    if (! $dryRun) {
                        DB::table('inside_pivots')->where('id', $pivot->id)->delete();
                    }

                    return;
                }
                /** @phpstan-ignore-next-line */
                $related = call_user_func($pivot->related_type.'::find', $pivot->related_uuid);
                if (! $related) {
                    $this->info(
                        "Le pivot {$pivot->id} a perdu sa relation et va être supprimé"
                    );
                    if (! $dryRun) {
                        $this->deletePivot($pivot->id);
                    }
                }
            }
        );
    }
}
