<?php

namespace Inside\Concerns;

use ErrorException;
use Exception;
use Illuminate\Contracts\Debug\ExceptionHandler;
use Inside\Exceptions\FatalThrowableError;
use Inside\Logging\LogManager;
use Monolog\Handler\NullHandler;
use Psr\Log\LoggerInterface;
use Symfony\Component\Console\Output\ConsoleOutput;
use Symfony\Component\ErrorHandler\Error\FatalError;
use Throwable;

trait ErrorHandling
{
    public static ?string $reservedMemory = null;

    protected function registerErrorHandling(): void
    {
        self::$reservedMemory = str_repeat('x', 32768);

        error_reporting(-1);

        set_error_handler($this->forwardsTo('handleError'));

        set_exception_handler($this->forwardsTo('handleException'));

        register_shutdown_function($this->forwardsTo('handleShutdown'));

        if (! $this->environment('testing')) {
            ini_set('display_errors', 'Off');
        }
    }

    protected function forwardsTo(string $method): callable
    {
        return fn (...$arguments) => $this->{$method}(...$arguments);
    }

    /**
     * @throws ErrorException
     */
    public function handleError(
        int $level,
        string $message,
        string $file = '',
        int $line = 0,
        array $context = []
    ): void {
        if ($this->levelErrorIsDeprecation($level)) {
            $this->handleDeprecationError($message, $file, $line, $level);

            return;
        }

        if (error_reporting() & $level) {
            throw new ErrorException($message, 0, $level, $file, $line);
        }
    }

    /**
     * @throws Exception
     */
    public function handleException(Throwable $exception): void
    {
        self::$reservedMemory = null;

        if (! $exception instanceof Exception) {
            $exception = new FatalThrowableError($exception);
        }

        try {
            $this->getExceptionHandler()->report($exception);
        } catch (Exception) {
        }

        if ($this->runningInConsole()) {
            $this->renderForConsole($exception);
        } else {
            $this->renderHttpResponse($exception);
        }
    }

    protected function renderForConsole(Exception $exception): void
    {
        $this->getExceptionHandler()->renderForConsole(new ConsoleOutput(), $exception);
    }

    /**
     * @throws Exception
     */
    protected function renderHttpResponse(Throwable $exception): void
    {
        if (! $exception instanceof Exception) {
            $exception = new FatalThrowableError($exception);
        }
        $this->getExceptionHandler()->render($this['request'], $exception)->send();
    }

    protected function getExceptionHandler(): ExceptionHandler
    {
        return $this->make(ExceptionHandler::class);
    }

    public function handleDeprecationError(string $message, string $file, int $line, int $level = E_DEPRECATED): void
    {
        if (! class_exists(LogManager::class)
            || ! $this->hasBeenBootstrapped()
            || $this->runningUnitTests()
        ) {
            return;
        }

        try {
            $logger = $this->make(LoggerInterface::class);
        } catch (Exception) {
            return;
        }

        $this->ensureDeprecationLoggerIsConfigured();

        $options = $this['config']->get('logging.deprecations') ?? [];

        with($logger->channel('deprecations'), function ($log) use ($message, $file, $line, $level, $options) {
            if ($options['trace'] ?? false) {
                $log->warning((string) new ErrorException($message, 0, $level, $file, $line));
            } else {
                $log->warning(sprintf('%s in %s on line %s', $message, $file, $line));
            }
        });
    }

    protected function ensureDeprecationLoggerIsConfigured(): void
    {
        with($this['config'], function ($config) {
            if ($config->get('logging.channels.deprecations')) {
                return;
            }

            $this->ensureNullLogDriverIsConfigured();

            if (is_array($options = $config->get('logging.deprecations'))) {
                $driver = $options['channel'] ?? 'null';
            } else {
                $driver = $options ?? 'null';
            }

            $config->set('logging.channels.deprecations', $config->get("logging.channels.{$driver}"));
        });
    }

    protected function ensureNullLogDriverIsConfigured(): void
    {
        with($this['config'], function ($config) {
            if ($config->get('logging.channels.null')) {
                return;
            }

            $config->set('logging.channels.null', [
                'driver' => 'monolog',
                'handler' => NullHandler::class,
            ]);
        });
    }

    public function handleShutdown(): void
    {
        self::$reservedMemory = null;

        if (! is_null($error = error_get_last()) && $this->typeErrorIsFatal($error['type'])) {
            $this->handleException($this->fatalErrorFromPhpError($error, 0));
        }
    }

    protected function levelErrorIsDeprecation(int $level): bool
    {
        return in_array($level, [E_DEPRECATED, E_USER_DEPRECATED]);
    }

    protected function typeErrorIsFatal(int $type): bool
    {
        return in_array($type, [E_COMPILE_ERROR, E_CORE_ERROR, E_ERROR, E_PARSE]);
    }

    protected function fatalErrorFromPhpError(array $error, ?int $traceOffset = null): FatalError
    {
        return new FatalError($error['message'], 0, $error, $traceOffset);
    }
}
