import DefaultTestE2E from '@maecia/inside-tests/e2e/class/DefaultTestE2E'
import moment from 'moment';

export default class ContentE2E extends DefaultTestE2E {
    constructor(props) {
        super(props)

        this.testName = 'Inside Content'
        this.faker = require('faker');
        this.faker.locale = 'fr' // TODO ..


        this.steps = [
            'start',
            'listSchema',
            'loadCurrentUser',
            'checkRandomContent',
            'checkContent',
            'createDocument',
            'createNews',
            'prepareSecurity',
            'reLoadCurrentUser',
            'loadPhpFile',
            'testSecurity',
            'cleanFinal'
        ]
        this.contents = []
        this.sections = []
        this.fakes = []
        this.user = null
        this.apiToken = null
        this.fakePhpFile = null
    }

    start() {
        it('Démarrage de session cypress via le back ( en super admin )', () => {
            cy.start('super_administrator', false)
        })
    }
    listSchema() {
        it('Récupération des schema', () => {
            cy.getHttp('/api/v1/schema').then((response) => {
                expect(response.status).to.eq(200)
                expect(response.body.data).to.be.not.empty

                Object.keys(response.body.data).forEach(e => {
                    let model = response.body.data[e]
                    if (model.class == undefined) return;
                    if (model.class.startsWith('Inside\\Content\\Models\\Contents\\')) {
                        this.contents.push(model)
                    } else {
                        this.sections.push(model)
                    }
                });
            })
        })
    }

    loadCurrentUser() {
        it('Récupération des informations de l\'utilisateur courant', () => {
            cy.authenticatedUser().then(o => {
                this.user = o
            })
        })
    }

    /**
     * Get random reference.s
     *
     * @param type
     * @param cardinality
     * @param locale
     * @returns {null}
     */
    getRandomReference(content, field, type, cardinality, required, locale) {
        if (type === undefined) {
            return null
        }
        //  cy.login() // TODO should not be necessary
        if (cardinality == 1) {
            if (!required && this.faker.random.boolean()) {
                return null;
            }
            cy.getHttp(`/api/v1/content/${type}`, {
                filters: {
                    lancode: locale,
                    offset: 0,
                    limit: 1,
                    sort: 'random'
                },
                fields: ['uuid']
            }).then((response) => {
                console.log(response)
                content[field] = response.body.data[0]['uuid']
            })
        } else {
            return cy.getHttp(`/api/v1/content/${type}`, {
                filters: {
                    lancode: locale,
                    offset: 0,
                    limit: this.faker.random.number({min: required ? 1 : 0, max: 6}),
                    sort: 'random'
                },
                fields: ['uuid']
            }).then((response) => {
                console.log(response)
                content[field] = response.body.data.map(o => o['uuid'])
            })
        }
    }

    /**
     * get random content data ( usefull to test PUT/POST on inside-content )
     *
     * @param contentDefinition content definition object
     * @returns {ObjectConstructor} new random content
     */
    generateRandomContent(contentDefinition) {
        if (this.fakes[contentDefinition['name']] === undefined) {
            this.fakes[contentDefinition['name']] = []
        }
        let content = {},
            locale = 'fr' // TODO get random local from available ones
        this.fakes[contentDefinition['name']].push(content)
        content['bundle'] = contentDefinition.name
        content['type'] = 'node'
        content['title'] = this.faker.lorem.sentence()
        content['langcode'] = locale
        content['author'] = this.user.uuid
        for (var field in contentDefinition.fields) {
            if (field == 'uuid') {
                content[field] = null
            }
            let definition = contentDefinition.fields[field]
            if (definition.id === undefined) {
                continue
            }

            switch (definition.type) {
                case 'list_float':
                case 'list_integer':
                case 'list_string':
                    content[definition['name']] = this.faker.random.arrayElement(definition['options']['allowed_values'][locale])
                    break
                case 'text':
                    content[definition['name']] = this.faker.lorem.sentence()
                    break
                case 'textarea':
                    content[definition['name']] = this.faker.lorem.text()
                    break
                case 'wysiwyg':
                    content[definition['name']] = this.faker.lorem.lines()
                    break
                case 'section':
                    // TODO
                    break
                case 'checkbox':
                    content[definition['name']] = this.faker.random.boolean()
                    break;
                case 'timestamp':
                    content[definition['name']] = moment(this.faker.date.past()).format('YYYY-MM-DD HH:mm:ss')
                    break;
                case 'reference':
                    content[definition['name']] = null
                    this.getRandomReference(content, definition['name'], definition['options']['target'][0], definition['options']['cardinality'], definition['options']['required'], locale)
                    break
                case 'image':
                    cy.image().then(path => {
                        content[definition['name']] = path
                    })
                    // TODO
                    break
                case 'file':
                    cy.file('pdf').then(path => {
                        content[definition['name']] = path
                    })
                    break
            }
        }
        return content
    }

    checkRandomContent() {
        it('Vérification de la création de données random', () => {
            let documentDefinition = this.contents.find(o => o['name'] === 'documents'),
                newsDefinition = this.contents.find(o => o['name'] === 'news')
            this.generateRandomContent(documentDefinition)
            this.generateRandomContent(newsDefinition)
            cy.factory('documents').then(content => {
                this.fakes['documents'].push(content)
            })
        })
    }

    createDocument() {
        it('Vérification de la création d\'un document', () => {
            cy.postHttp('/api/v1/content/documents', this.faker.random.arrayElement(this.fakes['documents'])).then((response) => {
                console.log(response)
                expect(response.status).to.eq(200)
            })
        })
    }

    createNews() {
        it('Vérification de la création d\'une actualité', () => {
            cy.postHttp('/api/v1/content/news', this.fakes['news'][0]).then((response) => {
                expect(response.status).to.eq(200)
            })
        })
    }

    checkContent() {
        it('Vérification des fakes', () => {
            console.log(this.fakes)
        })
    }

    prepareSecurity() {
        it('Préparation tests de sécurité', () => {
            cy.start('authenticated')
        })
    }

    reLoadCurrentUser() {
        it('Récupération des informations de l\'utilisateur courant', () => {
            // cy.login()
            cy.authenticatedUser().then(o => {
                this.user = o
            })
        })
    }

    loadPhpFile() {
        it('Prépare un fichier php', () => {
            cy.file('php').then(path => {
                this.fakePhpFile = path
                expect(this.fakePhpFile).not.to.be.empty
            })
            // On switch en francais pour avoir les messages de validation en francais
            cy.putHttp(`api/v1/content/users/${this.user.uuid}`, {
                langcode: 'fr'
            }).then((response) => {
                expect(response.status).to.eq(200)
            })
        })
    }

    testSecurity() {
        it('Tests de sécurité', () => {
            cy.log('Test d\'upload de photo avec un chemin /etc/passwd');
            cy.putHttp(`api/v1/content/users/${this.user.uuid}`, {
                image: '/etc/passwd'
            }).then((response) => {
                expect(response.status).to.eq(422)
                console.log(response.body.errors.image)
                expect(response.body.errors.image).to.have.lengthOf(4)
                expect(response.body.errors.image).to.include('Le fichier n\'existe pas.')
            })
            cy.putHttp(`api/v1/content/users/${this.user.uuid}`, {
                image: this.fakePhpFile
            }).then((response) => {
                expect(response.status).to.eq(422)
                console.log(response.body.errors.image)
                expect(response.body.errors.image).to.have.lengthOf(3)
                expect(response.body.errors.image).to.include('Le champ image doit être une image.')
            })
        })
    }

    cleanFinal() {
        it('Nettoyage de fin', () => {
            cy.cleanup().then((result) => {
                expect(result).to.not.equal(0)
            })
        })
    }
}
