<?php

declare(strict_types=1);

use Illuminate\Support\Facades\Route;
use Inside\Content\Http\Controllers\AdvancedTable\ExportController;
use Inside\Content\Http\Controllers\External\Index as ExternalIndex;
use Inside\Content\Http\Controllers\InsidePages\Create;
use Inside\Content\Http\Controllers\InsidePages\DeleteDraft;
use Inside\Content\Http\Controllers\InsidePages\Display;
use Inside\Content\Http\Controllers\InsidePages\Duplicate;
use Inside\Content\Http\Controllers\InsidePages\Get;
use Inside\Content\Http\Controllers\InsidePages\ListAllByPath;
use Inside\Content\Http\Controllers\InsidePages\Publish;
use Inside\Content\Http\Controllers\InsidePages\UploadFile;
use Inside\Content\Http\Controllers\Permissions\Can;
use Inside\Content\Http\Controllers\PublicPagesController;
use Inside\Content\Http\Controllers\Report;
use Inside\Content\Http\Controllers\Resource\Serve;
use Inside\Content\Http\Controllers\Resource\ServeContent;
use Inside\Content\Http\Controllers\Resource\ServeProtected;
use Inside\Content\Http\Controllers\Resource\ServePublic;
use Inside\Content\Http\Controllers\Resource\Upload;
use Inside\Content\Http\Controllers\Revision\Index;
use Inside\Content\Http\Controllers\Revision\Restore;
use Inside\Content\Http\Controllers\Revision\Serve as ServeRevision;
use Inside\Content\Http\Controllers\Revision\ServeFile;
use Inside\Routing\Router;

Route::group(
    [
        'namespace' => 'Inside\Content\Http\Controllers',
        'prefix' => 'api/v1',
        'middleware' => 'auth.api',
    ],
    function ($router) {
        // Wysiwyg images
        $router->post('content/wysiwyg/images', ['uses' => 'WysiwygImageController@store']);

        Route::group(
            [
                'middleware' => ['auth.backoffice.access:advanced-settings'],
            ],
            function ($router) {
                $router->post('content/asset', ['uses' => 'FileController@upload']);
                $router->delete('content/asset', ['uses' => 'FileController@destroy']);
            }
        );

        // Content
        $router->post('content/{type}', ['uses' => 'ContentController@create', 'as' => 'content.create']);
        $router->post('content/{type}/seen', ['uses' => 'UserContentSeenController']);
        $router->delete('content/{type}/{id}', ['uses' => 'ContentController@delete']);
        $router->delete('content/delete', ['uses' => 'ContentController@deleteMultiple']);
        $router->put('content/{type}/{id}', ['uses' => 'ContentController@update', 'as' => 'content.update']);
        $router->get('content/{types}', ['uses' => 'ContentController@list']);
        $router->post('content/{type}/export', ['uses' => 'ContentController@export']);
        $router->post('content/{type}/import', ['uses' => 'ContentController@import']);
        $router->get('content/{slug}/infos', ['uses' => 'ContentController@getSlugInfos']);
        $router->patch('content/collection/order', ['uses' => 'ContentCollectionOrderController']);

        //Events
        $router->post('participation/register', ['uses' => 'EventsController@register']);
        $router->delete('participation/unregister', ['uses' => 'EventsController@unregister']);
        $router->post('participation/export', ['uses' => 'EventsController@export']);

        // Form
        $router->get('schema', ['uses' => 'FormController@schema']);
        $router->get('list/form', ['uses' => 'FormController@listForm']);
        $router->get('form/{type}[/{id}]', ['uses' => 'FormController@form']);
        $router->get('field/{type}/{fieldName}', ['uses' => 'FormController@field']);
        $router->get('schema/{type}/fields', ['uses' => 'FormController@typeFields']);

        // Say yes
        $router->get('yes', ['uses' => 'YesController@yes']);
        $router->post('yes', ['uses' => 'YesController@yes']);
        $router->put('yes', ['uses' => 'YesController@yes']);
        $router->delete('yes', ['uses' => 'YesController@yes']);

        //Birthday
        $router->post('birthday/{toUser}', ['uses' => 'BirthdayController@notify']);
        $router->post('seniority_anniversary/{toUser}', ['uses' => 'SeniorityAnniversaryController@notify']);

        // Short urls
        $router->get('shorturl/{key}', 'ShortUrlController');

        // Calendar
        $router->post('export/event/calendar', ['uses' => 'CalendarController@exportCalendarEvent']);
        $router->post('export/invite/calendar', ['uses' => 'CalendarController@exportCalendarInvite']);

        // news categories
        $router->post('news_categories/follow', ['uses' => 'NewsCategoriesController@follow']);
        $router->delete('news_categories/unfollow', ['uses' => 'NewsCategoriesController@unfollow']);
        $router->get('news_categories/is-following', ['uses' => 'NewsCategoriesController@isFollowing']);
        $router->get('news_categories/following', ['uses' => 'NewsCategoriesController@following']);
        $router->get('news_categories/following-news', ['uses' => 'NewsCategoriesController@followingNews']);

        // content tree
        $router->get('content/tree/{type}/{id}', ['uses' => 'ContentTreeController@tree']);

        // Fonts
        Route::group([
            'prefix' => 'font',
        ],
            function (Router $router) {
                $router->get('google', 'GoogleFontsController');
            });
    }
);

// revisions
Route::group([
    'prefix' => 'api/v1',
    'middleware' => [
        'auth.api',
        'auth.role:super_administrator',
    ],
], function ($router) {
    $router->get('revision/{type}/{uuid}/{version}[/{baseVersion}]', Index::class);
    $router->post('revision/{type}/{uuid}/{version}', Restore::class);
});

// Api métier générique
Route::group([
    'namespace' => 'Inside\Content\Http\Controllers',
    'prefix' => 'api/v2',
    'middleware' => 'auth.api',
], function ($router) {
    $router->get('popin', ['uses' => 'PopinController@popin']);
    $router->post('reaction/popin', ['uses' => 'PopinController@reaction']);
    $router->get('weather/{uuid}', ['uses' => 'InsideWeatherController@getWeather']);
    $router->post('weather/cities', ['uses' => 'InsideWeatherController@getWeatherByCities']);
    $router->post('weather/research/city', ['uses' => 'InsideWeatherController@getLocationsForCity']);
    $router->post('weather/research/latlong', ['uses' => 'InsideWeatherController@getLocationsForLatlong']);
});

Route::group(
    [
        'middleware' => [
            //     'auth',
            //   'auth.role:super_administrator',
        ],
    ],
    function ($router) {
        $router->get(
            'images/revision/{type}/{uuid}/{version}/{path:.*}',
            [
                'uses' => ServeRevision::class,
                'as' => 'revision.serve',
            ]
        );
        $router->get(
            'files/revision/{type}/{uuid}/{version}/{path:.*}',
            [
                'uses' => ServeFile::class,
                'as' => 'revision.serve-file',
            ]
        );
    }
);

// External api content
Route::group(
    [
        'prefix' => 'api/v1',
    ],
    function ($router) {
        $router->get('external/{type:'.get_authorized_external_api_content_types().'}', ExternalIndex::class);
    }
);

// Resources
Route::group(
    [
        'middleware' => 'auth',
    ],
    function ($router) {
        $router->get('files/{slug}/{path:.*}', Serve::class);
        $router->get('protected_files/{path:.*}', ServeProtected::class);
    }
);

// Generate content routes
Route::group(
    [
        'prefix' => 'api/v1',
        'middleware' => 'auth.api',
    ],
    function ($router) {
        //Report
        $router->post('report', Report::class);
    }
);

Route::group(
    [
        'namespace' => 'Inside\Content\Http\Controllers',
        'middleware' => 'auth',
    ],
    function ($router) {
        $router->get('wysiwyg/images/{path:.*}', ['uses' => 'WysiwygImageController@serve']);
    }
);

// Special Unauthenticated partial access
Route::group(
    [
        'middleware' => 'auth.api',
    ],
    function ($router) {
        $router->get('images/{path:.*}', ServePublic::class);
        $router->get('content_files/{token}/{params:.*}', ServeContent::class);
    }
);

Route::group(
    [
        'prefix' => 'api/v1',
        'middleware' => 'auth.api',
    ],
    function ($router) {
        $router->post('file/upload', Upload::class);
    }
);

// Chat
Route::group(
    [
        'namespace' => 'Inside\Content\Chat\Http\Controllers',
        'prefix' => 'api/v1',
        'middleware' => 'auth.maecia',
    ],
    function ($router) {
        $router->get('messages', 'ChatController@fetchAllMessages');
        $router->post('messages', 'ChatController@sendMessage');
    }
);

// Public Pages
Route::group(
    ['prefix' => 'api/v1'],
    function ($router) {
        $router->get('public_pages/{slug}', PublicPagesController::class);
    }
);

// Permissions
Route::group(
    [
        'prefix' => 'api/v1',
        'middleware' => 'auth.api',
    ],
    function (Router $router) {
        $router->get('content/can/{action}/{contentType}/{uuid}', Can::class);
    }
);

// AdvancedTable Export
Route::group(
    [
        'prefix' => 'api/v1',
    ],
    function ($router) {
        $router->get('sections/advanced-table/{uuid}', ExportController::class);
    }
);

// inside pages
Route::group(
    [
        'prefix' => 'api/v1',
        'middleware' => [
            'auth.api',
        ],
    ],
    function ($router) {
        $router->get('inside-pages', Display::class);
    }
);

Route::group([
    'prefix' => 'api/v1',
    'middleware' => [
        'auth.role:super_administrator',
    ],
], function ($router) {
    $router->post('inside-pages', Create::class);
    $router->post('inside-pages/duplicate/{id}', Duplicate::class);
    $router->get('inside-pages/all', ListAllByPath::class);
    $router->post('inside-pages/upload', UploadFile::class);
    $router->get('inside-pages/{id}', Get::class);
    $router->delete('inside-pages/{id}', DeleteDraft::class);
    $router->put('inside-pages/{id}', Publish::class);
});
