<?php

namespace Inside\Content\Validation\Rules;

use Illuminate\Contracts\Validation\Rule;
use Illuminate\Support\Facades\Lang;

/**
 * Class IpOrRangeRule
 */
class IpOrRangeRule implements Rule
{
    /**
     * Determine if the validation rule passes.
     *
     * @param  string  $attribute
     * @param  mixed   $value
     * @return bool
     */
    public function passes($attribute, $value)
    {
        return $this->isValidIpOrRange($value);
    }

    /**
     * Get the validation error message.
     *
     * @return string
     */
    public function message()
    {
        return Lang::get('validation.invalid_ip_or_range');
    }

    /**
     * Check if the given value is a valid IP address or CIDR range.
     *
     * @param  string  $ipOrRange
     * @return bool
     */
    protected function isValidIpOrRange(string $ipOrRange): bool
    {
        if (filter_var($ipOrRange, FILTER_VALIDATE_IP)) {
            return true;
        }

        if (strpos($ipOrRange, '/') !== false) {
            [$base, $mask] = explode('/', $ipOrRange);

            return filter_var($base, FILTER_VALIDATE_IP) && is_numeric($mask) && $mask >= 1 && $mask <= 128;
        }

        return false;
    }
}
