<?php

namespace Inside\Content\Services;

use Exception;
use Illuminate\Database\Eloquent\Builder as EloquentBuilder;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Inside\Content\Contracts\WysiwygImageService;
use Inside\Content\Models\Content;
use Inside\Content\Models\WysiwygImage as WysiwygImageModel;
use Symfony\Component\HttpFoundation\File\MimeType\MimeTypeGuesser;

class WysiwygImage implements WysiwygImageService
{
    protected string $disk = 'wysiwyg_images';

    /**
     * @throws Exception
     */
    public function upload(UploadedFile $file, string $relativePath = '/'): string
    {
        $internal = $this->getSafeFileName(Str::random(64));
        $disk = Storage::disk($this->disk);

        // Find a new file name
        $internalPath = 'temp';
        while ($disk->exists($internalPath.$internal)) {
            $internal = $this->getSafeFileName(Str::random(64));
        }

        // Temporary save file
        $disk->putFileAs($internalPath, $file, $internal);
        $guesser = MimeTypeGuesser::getInstance();
        $image = WysiwygImageModel::create([
            'filename' => $this->getUniqueSafeFileName($file->getClientOriginalName()),
            'mimetype' => $guesser->guess($file->getRealPath()),
            'size' => filesize($file->getRealPath()),
            'disk' => $this->disk,
            'internal' => $internal,
            'internal_path' => $relativePath,
            'hash' => md5_file($file->getRealPath()),
            'imageable_type' => null,
            'imageable_id' => null,
        ]);

        return $this->getRelativePath($image->url);
    }

    public function store(Content $content, array $urls): bool
    {
        // Delete removed images
        foreach ($content->images as $image) {
            if (! in_array($this->getRelativePath($image->url), $urls)) {
                Log::debug(__(
                    '[WysiwygImage::store] removing old wysiwyg image ( front did not sent it ) <:url>  :type<:id>',
                    [
                        'url' => $image->url,
                        'type' => $image->imageable_type ?? '',
                        'id' => $image->imageable_id ?? '',
                    ]
                ),
                    [
                        'image' => $image,
                    ]
                );
                try {
                    $image->delete();
                } catch (Exception $exception) {
                    Log::debug(__(
                        '[WysiwygImage::store] failed to delete image with message :message!',
                        [
                            'message' => $exception->getMessage(),
                        ]
                    ),
                        [
                            'image' => $image,
                        ]
                    );
                }
            }
        }

        // Store new images
        $existingUrls = $content->images->pluck('url')->transform(function ($url) {
            return $this->getRelativePath($url);
        });

        $disk = Storage::disk($this->disk);

        foreach ($urls as $url) {
            if ($existingUrls->contains($url)) {
                continue;
            }
            $url = str_replace('/wysiwyg/images/', '', $url);
            [$year, $month, $fileName] = explode('/', $url, 3);

            // Find temp images
            /** @var WysiwygImageModel $image */
            try {
                /** @var WysiwygImageModel $image */
                $image = WysiwygImageModel::query()
                    ->whereYear('created_at', $year)
                    ->whereMonth('created_at', $month)
                    ->where('filename', $fileName)
                    ->firstOrFail();
            } catch (Exception) {
                continue;
            }

            /** When using duplicate button, the image myst be duplicated */
            if ($content->uuid !== $image->imageable_id) {
                /** @var WysiwygImageModel $duplicate */
                $duplicate = WysiwygImageModel::create(array_merge($image->toArray(), [
                    'imageable_type' => get_class($content),
                    'imageable_id' => $content->uuid,
                    'internal' => $this->getSafeFileName(Str::random(64)),
                ]));

                if ($disk->exists($image->base_path.$image->internal_path.$image->internal)) {
                    $disk->copy($image->base_path.$image->internal_path.$image->internal, $duplicate->base_path.$duplicate->internal_path.$duplicate->internal);
                }

                continue;
            }

            Log::debug(__(
                '[WysiwygImage::store] got uploaded wysiwyg image :filename attaching to content :type<:id>',
                [
                    'filename' => $fileName,
                    'type' => get_class($content),
                    'id' => $content->uuid ?? 'null',
                    'url' => $url,
                ]
            ),
                [
                    'image' => $image,
                ]
            );

            $oldPath = 'temp'.$image->internal_path.$image->internal;
            $image->update([
                'imageable_type' => get_class($content),
                'imageable_id' => $content->uuid,
            ]);
            if ($disk->exists($image->base_path.$image->internal_path.$image->internal)) {
                Log::error('Trying to add an existing image');
                continue;
            }
            $disk->move($oldPath, $image->base_path.$image->internal_path.$image->internal);
        }

        return true;
    }

    public function translate(Content $contentSource, Content $contentDestination, WysiwygImageModel $wysiwygImage): void
    {
        $translatedWysiwygImage = WysiwygImageModel::where([
            'imageable_type' => $wysiwygImage->imageable_type,
            'imageable_id' => $contentDestination->uuid,
            'hash' => $wysiwygImage->hash,
            'filename' => $wysiwygImage->filename,
        ])->first();

        if ($translatedWysiwygImage) {
            Log::debug(__(
                '[WysiwygImage::translate] :type wysiwyg image :filename is already translated for content :uuid',
                [
                    'filename' => $wysiwygImage->filename,
                    'type' => $wysiwygImage->imageable_type,
                    'uuid' => $contentDestination->uuid,
                ]
            ),
                [
                    'image' => $translatedWysiwygImage,
                ]
            );

            return;
        }

        $disk = Storage::disk($this->disk);
        $folder = class_to_type((string) $wysiwygImage->imageable_type);
        $imagePath = "/$folder/".$wysiwygImage->imageable_id.'/'.$wysiwygImage->internal;

        if ($disk->exists($imagePath)) {
            $internal = $this->getSafeFileName(Str::random(64));
            $translatedImagePath = "/$folder/".$contentDestination->uuid.'/'.$internal;
            $image = WysiwygImageModel::create([
                'filename' => $wysiwygImage->filename,
                'mimetype' => $wysiwygImage->mimetype,
                'size' => $wysiwygImage->size,
                'disk' => $wysiwygImage->disk,
                'internal' => $internal,
                'internal_path' => $wysiwygImage->internal_path,
                'hash' => $wysiwygImage->hash,
                'imageable_type' => $wysiwygImage->imageable_type,
                'imageable_id' => $contentDestination->uuid,
            ]);
            $disk->copy($imagePath, $translatedImagePath);
        }
    }

    /**
     * @throws Exception
     */
    public function load(string $path): ?WysiwygImageModel
    {
        [$year, $month, $fileName] = explode('/', $path, 3);
        $image = $this->sole(
            WysiwygImageModel::withoutGlobalScopes() // @phpstan-ignore-line
                ->whereYear('created_at', $year)
                ->whereMonth('created_at', $month)
                ->where('filename', $fileName)
        );

        return $image;
    }

    public function getRelativePath(string $fullUrl): string
    {
        return str_replace(config('app.url'), '', $fullUrl);
    }

    public function getExtensionFromFilename(string $name): string
    {
        if (false === ($dot = strrpos($name, '.'))) {
            return '';
        }
        $extension = substr($name, $dot + 1);

        if (str_contains($extension, '/')
            || (DIRECTORY_SEPARATOR === '\\' && str_contains($extension, '\\'))) {
            return '';
        }

        return $extension;
    }

    /**
     * @throws Exception
     */
    public function soleValue(EloquentBuilder $query, string $column): mixed
    {
        return $this->sole($query, [$column])->{\Inside\Support\Str::afterLast($column, '.')};
    }

    /**
     * @throws Exception
     */
    private function getSafeFileName(string $name, string $suffix = ''): string
    {
        if (! $name = $this->makeSafe($name)) {
            throw new Exception('failed to make file name safe !!');
        }

        $name = punycode_encode($name);
        $extension = $this->getExtensionFromFilename($name);
        if (! empty($extension)) {
            $extension = '.'.Str::lower($extension);
        }

        $nameWithoutExtension = substr($name, 0, strlen($name) - strlen($extension));

        return $nameWithoutExtension.$suffix.$extension;
    }

    /**
     * @throws Exception
     */
    private function getUniqueSafeFileName(string $name): string
    {
        $safeFileName = $this->getSafeFileName($name);
        $i = 1;
        while (WysiwygImageModel::where(
            'filename',
            $safeFileName
        )->exists()) {
            $safeFileName = $this->getSafeFileName($name, '_'.$i++);
        }

        return $safeFileName;
    }

    /**
     * @throws Exception
     */
    private function sole(EloquentBuilder $query, array $columns = ['*']): mixed
    {
        $result = $query->take(2)->get($columns);
        $count = $result->count();

        if ($count === 0) {
            throw (new ModelNotFoundException())->setModel(WysiwygImageModel::class);
        }

        if ($count > 2) {
            throw new Exception('[WysiwygImage::sole] Wrong count => '.$count);
        }

        return $result->last();
    }

    private function makeSafe(string $name): string
    {
        $name = rtrim($name, '.');
        if (function_exists('transliterator_transliterate') && function_exists('iconv')) {
            $transliterate = transliterator_transliterate('Any-Latin; Latin-ASCII', $name);
            if ($transliterate) {
                /** @var string $name */
                $name = iconv(
                    'UTF-8',
                    'ASCII//TRANSLIT//IGNORE',
                    $transliterate
                );
            }
        }

        $regex = ['#(\.){2,}#', '#[^A-Za-z0-9\.\_\- ]#', '#^\.#'];

        return (string) preg_replace('/\s+/', '_', trim((string) preg_replace($regex, '', $name)));
    }
}
