<?php

declare(strict_types=1);

namespace Inside\Content\Services\WelcomeToTheJungle;

use Exception;
use Illuminate\Support\Facades\Lang;
use Inside\Content\Contracts\WelcomeToTheJungleContentsManager as WelcomeToTheJungleContentsManagerContract;
use Inside\Content\Facades\ContentBlueprints;
use Inside\Content\Models\Contents\Forms;
use Inside\Content\Models\Contents\JobOffers;
use Inside\Host\Bridge\BridgeContent;

final class WelcomeToTheJungleContentsManager implements WelcomeToTheJungleContentsManagerContract
{
    private bool $isEnabled;

    private BridgeContent $bridge;

    public function __construct()
    {
        $this->isEnabled = ContentBlueprints::isBundleInstalled('wttj');
    }

    public function getBridge(): BridgeContent
    {
        return $this->bridge ??= new BridgeContent();
    }

    public function isWTTJEnabled(): bool
    {
        return $this->isEnabled;
    }

    public function createOrUpdateJobOffer(array $jobOfferData, ?string $langcode = null): ?string
    {
        if (! $this->isEnabled) {
            throw new Exception('WelcomeToTheJungle module is not enabled !');
        }

        $langcode ??= config('app.locale');

        $existingOffer = JobOffers::query()
            ->where('langcode', $langcode)
            ->where('external_reference', $jobOfferData['external_reference'])
            ->first();

        $data = array_filter([
            'title' => $jobOfferData['title'],
            'external_reference' => $jobOfferData['external_reference'],
            'external_link' => $jobOfferData['external_link'],
            'locations' => $this->getReference('locations', $jobOfferData['location_title'], $langcode),
            'contracts' => $this->getReference('contracts', $jobOfferData['contract_title'], $langcode),
            'wttj_applications' => $this->getReference(
                'wttj_applications',
                $jobOfferData['organization_reference'],
                $langcode,
                'wttj_reference',
                false
            ),
            'content' => array_filter([
                $this->formatParagraph('description', $jobOfferData['description'], $langcode),
                $this->formatParagraph('profile', $jobOfferData['profile'], $langcode),
                $this->formatParagraph('recruitment_process', $jobOfferData['recruitment_process'], $langcode),
            ]),
            'forms' => $this->createOrUpdateAssociatedForm($jobOfferData['title'], $existingOffer?->forms->first()?->uuid, $langcode),
            'uuid' => $existingOffer?->uuid,
            'langcode' => $langcode,
            'status' => true,
        ]);

        return $this->getBridge()->contentInsert('job_offers', $data, ! isset($data['uuid']));
    }

    private function getReference(string $contentType, string $value, string $langcode, string $identifier = 'title', bool $create = true): ?string
    {
        $existing = type_to_class($contentType)::query()
            ->where($identifier, $value)
            ->where('langcode', $langcode)
            ->first()?->uuid;

        if (! $create) {
            return $existing;
        }

        return $existing ?? $this->getBridge()->contentInsert($contentType, [$identifier => $value]);
    }

    private function createOrUpdateAssociatedForm(string $title, ?string $existingForm, string $langcode): ?string
    {
        $formTemplate = Forms::query()
            ->where('job_offers_form_template', true)
            ->where('langcode', $langcode)
            ->first();

        if (! $formTemplate instanceof Forms) {
            throw new Exception("A base form should have been created (langcode: $langcode)");
        }

        $data = [
            'title' => Lang::get('job_offers.wttj.form_application.title', ['title' => $title], $langcode),
            'form_type' => $formTemplate->form_type,
            'show_trends_user' => $formTemplate->show_trends_user,
            'only_one_reply' => $formTemplate->only_one_reply,
            'anonym' => $formTemplate->anonym,
            'show_my_replies' => $formTemplate->show_my_replies,
            'has_status' => $formTemplate->has_status,
            'inputs' => $formTemplate->inputs,
            'aside_desc_admin' => $formTemplate->aside_desc_admin,
            'aside_desc_user' => $formTemplate->aside_desc_user,
            'admin_roles' => $formTemplate->admin_roles,
            'forms_introduction' => $formTemplate->forms_introduction,
            'job_offers_form' => true,
            'langcode' => $langcode,
        ];

        if ($existingForm) {
            $data['uuid'] = $existingForm;
        }

        return $this->getBridge()->contentInsert('forms', $data, ! isset($data['uuid']));
    }

    private function formatParagraph(string $name, ?string $value, string $langcode): ?array
    {
        if (empty($value)) {
            return null;
        }

        $title = match ($name) {
            'description', 'profile', 'recruitment_process' => Lang::get("job_offers.wttj.form_application.paragraph.$name", [], $langcode),
            default => null
        };

        if (! empty($title)) {
            $value = "<h2>$title</h2><br/>$value";
        }

        return [
            'bundle' => 'text',
            'body' => $value,
        ];
    }

    public function updateJobOffersApplicationForms(string $langcode): void
    {
        if (! $this->isEnabled) {
            return;
        }

        $template = Forms::query()
            ->where('job_offers_form_template', true)
            ->where('langcode', $langcode)
            ->first();

        $jobOffersForms = Forms::query()
            ->where('job_offers_form', true)
            ->where('langcode', $langcode)
            ->get();

        if (! $template instanceof Forms || $jobOffersForms->isEmpty()) {
            return;
        }

        $jobOffersForms->each(function (Forms $form) use ($template) {
            $this->getBridge()->contentUpdate($form->content_type, [
                'uuid' => $form->uuid,
                'form_type' => $template->form_type,
                'show_trends_user' => $template->show_trends_user,
                'only_one_reply' => $template->only_one_reply,
                'anonym' => $template->anonym,
                'show_my_replies' => $template->show_my_replies,
                'has_status' => $template->has_status,
                'inputs' => $template->inputs,
                'aside_desc_admin' => $template->aside_desc_admin,
                'aside_desc_user' => $template->aside_desc_user,
                'admin_roles' => $template->admin_roles,
                'forms_introduction' => $template->forms_introduction,
                'job_offers_form' => true,
            ]);
        });
    }

    public function deleteOrphanJobOffersApplicationForms(): void
    {
        if (! $this->isEnabled) {
            return;
        }

        $orphanForms = Forms::where('job_offers_form', true)
            ->get()
            ->filter(fn (Forms $form) => $form->reverseJobOffers->isEmpty());

        if ($orphanForms->isEmpty()) {
            return;
        }

        $orphanForms->each(function (Forms $form) {
            if ($form->uuid) {
                $this->getBridge()->contentDelete($form->content_type, $form->uuid);
            }
        });
    }

    public function updateJobOfferApplicationFormTitle(string $jobOfferUuid): void
    {
        if (! $this->isEnabled) {
            return;
        }

        $jobOffer = JobOffers::find($jobOfferUuid);

        if (! $jobOffer instanceof JobOffers || $jobOffer->forms->isEmpty()) {
            return;
        }

        $this->getBridge()->contentUpdate('forms', [
            'uuid' => $jobOffer->forms->first()->uuid,
            'title' => Lang::get('job_offers.wttj.form_application.title', ['title' => $jobOffer->title], $jobOffer->langcode),
        ]);
    }
}
