<?php

namespace Inside\Content\Services;

use Exception;
use Illuminate\Support\Facades\Gate;
use Inside\Authentication\Models\User;
use Inside\Content\Facades\Schema;
use Inside\Content\Models\Content;
use Inside\Content\Scopes\PublishedScope;
use Inside\Permission\Exodus\Enums\CapabilityEnum;
use Inside\Permission\Facades\Permission;

class ScopeLogicService
{
    /**
     * @throws Exception
     */
    public function statusScopeNeeded(User $user, string $type): bool
    {
        $bypassStatusScopeCallbacks = config('contents.bypass_status_scope_callbacks', []);
        foreach ($bypassStatusScopeCallbacks as $callback) {
            if (is_callable($callback)) {
                if ($callback($type) === true) {
                    return false;
                }
            }
        }

        $specialRoles = array_unique(array_merge(config('permission.special_roles', []), ['super_administrator']));
        $permissionUser = $user->permission;

        if ($permissionUser->hasAnyRole($specialRoles)) {
            return false;
        }

        // If user has access to backoffice and we access to users content, we can see non published content
        if ($type == 'users' && Gate::forUser($user)->allows('access_user') || Gate::forUser($user)->allows('access_statistic')) {
            return false;
        }

        // If user can create content, it should be able to see non published content
        if (Permission::isSystemV2Enabled()) {
            if (Schema::isContentType($type) && $user->hasContentTypePrivilegeTo(CapabilityEnum::CREATE, type_to_class($type))) {
                return false;
            }
        } else {
            if ($type === 'users' || Permission::allowed('create', $type, null, $user)) {
                return false;
            }
        }

        return true;
    }

    /**
     * @throws Exception
     */
    public function publishabledScopeNeeded(User $user, string $type): bool
    {
        $bypassPublishableScopeCallbacks = config('contents.bypass_publishable_scope_callbacks', []);
        foreach ($bypassPublishableScopeCallbacks as $callback) {
            if (is_callable($callback)) {
                if ($callback($type) === true) {
                    return false;
                }
            }
        }

        $specialRoles = array_unique(array_merge(config('permission.special_roles', []), ['super_administrator']));
        $permissionUser = $user->permission;

        if ($permissionUser->hasAnyRole($specialRoles)) {
            return false;
        }

        // If user can create content, it should be able to see non published content
        if (Permission::isSystemV2Enabled()) {
            if (Schema::isContentType($type) && $user->hasContentTypePrivilegeTo(CapabilityEnum::CREATE, type_to_class($type))) {
                return false;
            }
        } else {
            if (Permission::allowed('create', $type, null, $user)) {
                return false;
            }
        }

        return true;
    }
}
