<?php

declare(strict_types=1);

namespace Inside\Content\Services\Queries;

use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Str;
use Inside\Content\Contracts\DrupalSchema as DrupalSchemaContract;
use Inside\Content\Facades\Schema;
use Inside\Content\Services\Schema\SchemaService;
use Inside\Facades\Package;
use Inside\Reaction\Models\Reaction;
use Inside\Statistics\Models\Statistic;

final class DrupalSchema implements DrupalSchemaContract
{
    /**
     * Request for the content schema.
     */
    public function __invoke(): array
    {
        $schema = [];
        foreach (Schema::models() ?? [] as $type => $modelOptions) {
            $class = $modelOptions['type'] == SchemaService::CONTENT_MODEL_TYPE
                ? type_to_class($type)
                : section_type_to_class($type);
            $schema[$class] = Schema::formatModelToLegacyFront($type, $modelOptions);

            if (! isset($schema[$class]['fields'])) {
                $schema[$class]['fields'] = [];
            }
            foreach ($modelOptions['fields'] as $fieldName => $fieldOptions) {
                $schema[$class]['fields'][$fieldName] = Schema::formatFieldToLegacyFront($fieldName, $fieldOptions);
            }
            $this->addCustomFields($schema[$class]['fields'], $type);
        }

        $this->reverseReferences($schema);

        if (count($schema) == 0) {
            return [];
        }

        $schema[Reaction::class] = [
            'model' => Reaction::class,
            'options' => [
                'name' => 'reactions',
                'translatable' => false,
                'title' => $this->getTranslations('field.reactions'),
            ],
            'fields' => [
                'users' => [
                    'name' => 'users',
                    'type' => 'reference',
                    'target' => ['users'],
                    'options' => [
                        'weight' => 100,
                        'title' => $this->getTranslations('field.authors'),
                    ],
                ],
                'count' => [
                    'name' => 'count',
                    'type' => 'onoff',
                    'options' => [
                        'title' => $this->getTranslations('field.count'),
                        'weight' => 1000,
                    ],
                ],
            ],
        ];

        $schema[Statistic::class] = [
            'model' => Statistic::class,
            'options' => [
                'name' => 'statistics',
                'translatable' => false,
                'title' => $this->getTranslations('field.statistics'),
            ],
            'fields' => [
                'users' => [
                    'name' => 'users',
                    'type' => 'reference',
                    'target' => ['users'],
                    'options' => [
                        'weight' => 100,
                        'title' => $this->getTranslations('field.authors'),
                    ],
                ],
                'count' => [
                    'name' => 'count',
                    'type' => 'onoff',
                    'options' => [
                        'title' => $this->getTranslations('field.count'),
                        'weight' => 1000,
                    ],
                ],
            ],
        ];

        return $schema;
    }

    private function getTranslations(string $key): array
    {
        return collect(list_languages())
            ->mapWithKeys(fn ($langcode) => [$langcode => Lang::get($key, [], $langcode)])
            ->all();
    }

    /**
     * Reverse reference fields
     *
     * @param array $schema
     * @return void
     */
    protected function reverseReferences(array &$schema)
    {
        foreach ($schema as $className => $modelOptions) {
            if (! isset($modelOptions['fields'])) {
                return;
            }

            if ($modelOptions['type'] === SchemaService::SECTION_MODEL_TYPE) {
                continue; // Do not allow reverse on section
            }

            $protectedFields = ['authors', 'reactions', 'statistics'];

            foreach ($modelOptions['fields'] as $fieldName => $fieldOptions) {
                if (! is_array($fieldOptions) || ! isset($fieldOptions['type'])) {
                    continue;
                }

                if (
                    ($fieldOptions['type'] == 'reference' ||
                        $fieldOptions['type'] == 'comment') &&
                    ! in_array($fieldName, $protectedFields)
                ) {
                    if (isset($fieldOptions['options']['target'])) {
                        $bundle = str_replace('Inside\Content\Models\Sections\\', '', $className);
                        $bundle = Str::snake(str_replace('Inside\Content\Models\Contents\\', '', $bundle));

                        $schema[$className]['fields'][$fieldName]['target'] = $fieldOptions['options']['target'];

                        foreach ((array) $fieldOptions['options']['target'] as $target) {
                            $referencedFieldNames = Schema::getFieldNamesThatReferenceType($target);
                            $reverseSchemaField = $fieldSchema = [
                                'type' => 'reference',
                                'model' => 'Inside\Content\Models\Contents\\'.Str::studly($target),
                                'target' => [$bundle],
                                'name' => $bundle,
                                'reverse' => true,
                                'field_name' => $fieldName,
                                'options' => [
                                    'title' => $bundle,
                                    'target' => $referencedFieldNames[$bundle] ?? [],
                                    'weight' => 0,
                                    'cardinality' => -1,
                                ],
                            ];
                            if (
                                ! isset(
                                    $schema['Inside\Content\Models\Contents\\'.
                                    Str::studly($target)]['fields'][$bundle]
                                )
                            ) {
                                $schema['Inside\Content\Models\Contents\\'.Str::studly($target)]['fields'][$bundle] =
                                    $fieldSchema;
                            }

                            // Adding reverse by field
                            if (
                                ! isset(
                                    $schema['Inside\Content\Models\Contents\\'.
                                    Str::studly($target)]['fields'][$bundle.'.'.$fieldName]
                                )
                            ) {
                                $reverseSchemaField['name'] = $bundle.'.'.$fieldName;
                                $reverseSchemaField['options']['target'] = [$fieldName];
                                $schema['Inside\Content\Models\Contents\\'.
                                Str::studly($target)]['fields'][$bundle.'.'.$fieldName] = $reverseSchemaField;
                            }
                        }
                    }
                }
            }
        }
    }

    /**
     * @param array $schema
     * @param string $type
     * @return void
     */
    protected function addCustomFields(array &$schema, string $type): void
    {
        $typeOptions = Schema::getModelOptions($type);

        $customs = [
            ['name' => 'uuid', 'type' => 'string', 'title' => Lang::get('field.uuid')],
            ['name' => 'count', 'type' => 'onoff', 'title' => Lang::get('field.count')],
            ['name' => 'total_count', 'type' => 'onoff', 'title' => Lang::get('field.total_count')],
            ['name' => 'total', 'type' => 'onoff', 'title' => Lang::get('field.total')],
            ['name' => 'content_type', 'type' => 'string', 'title' => Lang::get('field.content_type')],
            ['name' => 'tree', 'type' => 'onoff', 'title' => Lang::get('field.tree')],
            ['name' => 'pid', 'type' => 'string', 'title' => Lang::get('field.pid')],
            ['name' => 'admin', 'type' => 'onoff', 'title' => Lang::get('field.admin')],
            ['name' => 'created_at', 'type' => 'timestamp', 'title' => Lang::get('field.created_at')],
            ['name' => 'updated_at', 'type' => 'timestamp', 'title' => Lang::get('field.updated_at')],
            ['name' => 'published_at', 'type' => 'timestamp', 'title' => Lang::get('contents.fields.published_at.title')],
            [
                'name' => 'parent_tree',
                'type' => 'onoff',
                'title' => Lang::get('field.parent_tree'),
                'target' => [],
            ],
            [
                'name' => 'reactions',
                'type' => 'reference',
                'title' => Lang::get('field.reactions'),
                'target' => ['reactions'],
            ],
            [
                'name' => 'authors',
                'type' => 'reference',
                'title' => Lang::get('field.authors'),
                'target' => ['users'],
            ],
        ];

        // isAliasable / slug
        if (isset($typeOptions['aliasable']) && $typeOptions['aliasable']) {
            $customs[] = ['name' => 'slug', 'type' => 'string', 'title' => Lang::get('field.url')];
        }

        // has section
        $hasSection = [
            'name' => 'has_section',
            'type' => 'custom_with_fields',
            'title' => Lang::get('field.has_section'),
            'options' => [
                'fields' => Schema::getFieldListing(
                    $type,
                    function ($options) {
                        return $options['type'] === 'section';
                    }
                ),
            ],
        ];
        if (! empty($hasSection['options']['fields'])) {
            $customs[] = $hasSection;
        }

        $customs[] = [
            'name' => 'statistics',
            'type' => 'reference',
            'title' => Lang::get('field.statistics'),
            'target' => ['statistics'],
        ];

        $customs[] = [
            'name' => 'unique_views',
            'type' => 'onoff',
            'title' => Lang::get('field.unique_views'),
        ];

        $customs[] = [
            'name' => 'total_views',
            'type' => 'onoff',
            'title' => Lang::get('field.total_views'),
        ];

        if (Package::has('inside-archive')) {
            $customs[] = [
                'name' => 'archived',
                'type' => 'onoff',
                'title' => Lang::get('field.archived'),
            ];
        }

        if (isset($schema['author'])) {
            unset($schema['author']);
        }

        if (isset($schema['update_author'])) {
            unset($schema['update_author']);
        }

        if (isset($schema['published_at'])) {
            unset($schema['published_at']);
        }

        foreach ($customs as $custom) {
            $field = [
                'name' => $custom['name'],
                'type' => $custom['type'],
                'options' => [
                    'title' => $custom['title'],
                    'weight' => 1000,
                ],
            ];
            if (isset($custom['options'])) {
                $field['options'] = array_merge($custom['options'], $field['options']);
            }

            if ($custom['type'] == 'reference') {
                $field['target'] = $custom['target'];
                $field['options']['target'] = $custom['target'];
            }

            if (! isset($schema[$custom['name']])) {
                $schema[$custom['name']] = $field;
            }
        }
    }
}
