<?php

declare(strict_types=1);

namespace Inside\Content\Services\Queries;

use Exception;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Http\Request;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Schema;
use Inside\Content\Exceptions\FieldSchemaNotFoundException;
use Inside\Content\Facades\ContentHelper;
use Inside\Content\Facades\Schema as InsideSchema;
use Inside\Content\Models\Content;
use Inside\Content\Services\Queries\Traits\DateFilterQuery;
use Inside\Content\Services\Queries\Traits\DateSortQuery;
use Inside\Content\Transformers\ContentTransformer;
use Inside\Database\Eloquent\Builder;
use Inside\Facades\Package;
use Inside\Statistics\Events\AnalyticViewLogEvent;
use Inside\Statistics\Facades\Stats;
use Inside\Support\Str;

final class ContentQuery
{
    use DateFilterQuery;
    use DateSortQuery;

    /**
     * The requested fields
     */
    protected array $fields = [];

    /**
     * The requested filters
     */
    protected array $filters = [];

    /**
     * The requested types
     */
    protected array $types = [];

    /**
     * The requested options
     */
    protected array $options = [];

    /**
     * The request url
     */
    protected string|null $path;

    /**
     * The current page
     */
    protected int $page = 1;

    /**
     * The content transformer
     */
    protected ContentTransformer $transformer;

    /**
     * The result collection
     */
    protected Collection|LengthAwarePaginator $collection;

    /**
     * List of authorized path
     *
     * @var array<int, string>
     */
    protected array $authorizedPaths = [
        'api/v1/public',
    ];

    /**
     * The callback query
     */
    private string $queryCallback = 'query';

    /**
     * @var array
     */
    private array $userContentSeenTypes;

    /**
     * Create a new query content instance.
     */
    public function __construct()
    {
        $this->collection = new Collection();
        $this->transformer = new ContentTransformer();
        $this->userContentSeenTypes = config('user-seen');
    }

    /**
     * Extract fields input
     */
    protected function extractFieldsInput(Request $request): array
    {
        $this->fields = ContentHelper::extractFieldsInputFromRequest($request);

        return $this->fields;
    }

    /**
     * Extract filters input
     */
    protected function extractFiltersInput(string $type, Request $request): array
    {
        $this->filters = ContentHelper::extractFiltersInputFromRequest($request);

        /**
         * The author is a normal field in the content tables
         */
        if (isset($this->filters['authors'], $this->filters['authors']['uuid:eq'])) {
            $this->filters['author'] = $this->filters['authors']['uuid:eq'];
            unset($this->filters['authors']);
        }

        if ($type == 'users') {
            $this->filters = collect($this->filters)
                ->reject(fn ($value, $key) => str_starts_with($key, 'published_at:'))
                ->all();
        }

        foreach ($this->filters as $key => &$filter) {
            if (is_array($filter) && (isset($filter['uuid:in']) && is_array($filter['uuid:in']))) {
                $types = explode(',', $type);
                $class = get_reference_class($types[0], $key);
                if ($class === null) {
                    warn_front('Can not find reference class ['.$types[0].'] ('.$key.')');
                    Log::warning('Can not find reference class ['.$types[0].'] ('.$key.')');
                    continue;
                }

                $uuids = [];

                foreach ($filter['uuid:in'] as $uuid) {
                    $uuids[] = $uuid;
                    $model = call_user_func($class.'::find', $uuid);

                    if (empty($model)) {
                        continue;
                    }

                    $query = call_user_func($class.'::query');
                    $translations =
                      $query->where('uuid_host', $model->uuid_host)->where('langcode', '<>', $model->langcode)->get()
                        ->pluck('uuid')->toArray();

                    if (count($translations)) {
                        $uuids = array_merge($uuids, $translations);
                    }
                }

                $filter['uuid:in'] = $uuids;
            }

            if (is_array($filter)
              && ((isset($filter['uuid']) && ! is_array($filter['uuid']))
                || (isset($filter['uuid:eq']) && ! is_array($filter['uuid:eq'])))
            ) {
                $filterKey = isset($filter['uuid']) ? 'uuid' : 'uuid:eq';

                $types = explode(',', $type);
                $class = get_reference_class($types[0], $key);
                if (is_null($class)) {
                    warn_front('Wrong filter on reference field ['.$key.'] for class ['.$types[0].']');
                    Log::warning('Wrong filter on reference field ['.$key.'] for class ['.$types[0].']');
                    continue;
                }
                $model = call_user_func($class.'::find', $filter[$filterKey]);

                if (empty($model)) {
                    continue;
                }

                $query = call_user_func($class.'::query');
                $translations = $query->where('uuid_host', $model->uuid_host)->get()->pluck('uuid')->toArray();

                if (count($translations)) {
                    $filter['uuid:in'] = $translations;
                    unset($filter[$filterKey]);
                }
            }

            if (is_array($filter) && isset($filter['root']) && ! is_array($filter['root'])) {
                $class = 'Inside\\Content\\Models\\Contents\\'.Str::studly($key);

                $model = call_user_func($class.'::find', $filter['root']);

                $uuids = [$filter['root']];

                $this->getChildren($model, $key, $uuids);
                $filter['uuid:in'] = $uuids;
                unset($filter['root']);
            }
        }

        $this->transformer->setFilters($this->filters);

        return $this->filters;
    }

    private function getChildren(Content $model, string $type, array &$uuids): void
    {
        $children = $model->belongsToMany(
            type_to_class($type),
            'inside_pivots',
            'related_uuid',
            'parent_uuid'
        )->wherePivot('related_langcode', $model->langcode)
          ->withPivot('weight')
          ->orderBy('pivot_weight')->get();

        foreach ($children as $child) {
            $uuids[] = $child->uuid;

            $this->getChildren($child, $type, $uuids);
        }
    }

    public function isRandom(): bool
    {
        $sort = $this->filters['sort'] ?? null;

        return $sort === 'random';
    }

    /**
     * Request content by HTTP request
     *
     * @param  string  $types
     * @param  Request  $request
     * @return Collection|LengthAwarePaginator|\Illuminate\Support\Collection
     */
    public function requestQuery(string $types, Request $request)//: Collection
    {
        $this->extractFieldsInput($request);
        $this->extractFiltersInput($types, $request);

        $this->path = $request->url();
        $this->isAuthorized();

        if ($request->has('page')) {
            $this->page = intval($request->get('page'));
        }

        /**
         * _contents is a special content type
         * it means all front administrable contents
         */
        if ($types == '_contents') {
            $filteredContentTypes = null;
            if (isset($this->filters['content_type'])) {
                $filteredContentTypes = $this->filters['content_type'];
                if (! is_array($filteredContentTypes)) {
                    $filteredContentTypes = [$filteredContentTypes];
                }
            }

            if ($filteredContentTypes !== null) {
                $this->types = array_map('trim', $filteredContentTypes);
            } else {
                // Magic content type
                $this->types = InsideSchema::getContentTypes(
                    function ($model) {
                        if (! isset($model['options']['name'])) {
                            return false;
                        }
                        $name = $model['options']['name'];

                        // Remove comments, users and menus
                        return ! Str::endsWith($name, '_menus') && ! in_array($name, ['users', 'comments']);
                    }
                );
            }
        } else {
            $this->types = explode(',', $types);
        }

        $this->types = array_filter(
            $this->types,
            function ($type) {
                return InsideSchema::hasModel($type);
            }
        );

        return $this->doQuery();
    }

    /**
     * Find element
     *
     * @param  string  $type
     * @param  string  $uuid
     * @param  Request  $request
     * @return Collection
     */
    public function find(string $type, string $uuid, Request $request): Collection
    {
        $this->extractFieldsInput($request);
        $this->extractFiltersInput($type, $request);

        $this->path = $request->url();
        $this->isAuthorized();

        $query = call_user_func('\Inside\Content\Models\Contents\\'.Str::studly($type).'::'.$this->queryCallback);
        $query->where('uuid', $uuid);

        // Statistics fields
        $this->addStatisticsFieldsToQuery($type, $query);

        $this->collection = $query->select(type_to_table($type).'.*')->get();

        return $this->collection;
    }

    /**
     * @param  string  $type
     * @param  Builder  $query
     * @return void
     */
    protected function addStatisticsFieldsToQuery(string $type, Builder $query)
    {
        if (in_array('unique_views', $this->fields)) {
            Stats::addUniqueViewsToQuery($type, $query);
        }
        if (in_array('total_views', $this->fields)) {
            Stats::addTotalViewsToQuery($type, $query);
        }
    }

    /**
     * Sort result
     *
     * @return void
     */
    protected function applyCollectionSort()
    {
        if (! isset($this->filters['sort'])) {
            return;
        }

        if ($this->filters['sort'] == 'random') {
            if ($this->collection instanceof Collection) {
                $this->collection = $this->collection->shuffle();
            }
        } else {
            if (! is_array($this->filters['sort'])) {
                $this->filters['sort'] = [$this->filters['sort']];
            }

            foreach (array_reverse($this->filters['sort']) as $sort) {
                $filter = explode(':', $sort);
                if (count($filter) > 1) {
                    $descending = $filter[1] == 'desc';
                    if ($this->collection instanceof Collection) {
                        $filterWith = $filter[0];

                        $this->collection = $this->collection->sortBy($filterWith, SORT_NATURAL | SORT_FLAG_CASE, $descending);

                        $this->collection = $this->collection
                          ->sortBy($filterWith, SORT_STRING | SORT_FLAG_CASE | SORT_NATURAL, $descending)
                          // we remove non-case sensitive words to have non-words char first (like "[")
                          ->filter(fn (mixed $item) => Str::contains($filterWith, 'archive.') ? true : preg_match('/^\W/', (string) $item->$filterWith))
                          // we concat non-case sensitive words after the non-words.
                          ->concat($this->collection->filter(fn (mixed $item) => preg_match('/^\w/', (string) $item->$filterWith)));
                    }
                }
            }
        }
    }

    /**
     * Paginate results
     *
     * @return void
     */
    protected function applyCollectionPaginate()
    {
        if (isset($this->filters['limit'])) {
            $total = count($this->collection);

            $offset = $this->filters['offset'] ?? 0;
            if ($this->page > 1) {
                $offset += ($this->page - 1) * $this->filters['limit'];
            }

            if ($this->collection instanceof Collection) {
                $this->collection = $this->collection->splice($offset, $this->filters['limit']);
            }

            $this->collection = new LengthAwarePaginator(
                $this->collection,
                $total,
                $this->filters['limit'],
                $this->page,
                ['path' => $this->path]
            );
        }
    }

    /**
     * Apply relation
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query  current query
     * @param  string  $fieldName  field for the relation
     * @param  array  $filters  filters to apply to the field
     * @param  string  $type  type of content on which apply relation
     * @return void
     */
    protected function applyRelation(
    \Illuminate\Database\Eloquent\Builder $query,
    string $fieldName,
    array $filters,
    string $type
  ) {
        if ($type == 'comments') {
            // Note comments does not have any relation fields it's a trick
            $type = $fieldName;
        } else {
            try {
                $fieldOptions = InsideSchema::getFieldOptions($type, $fieldName);
            } catch (FieldSchemaNotFoundException $exception) {
                warn_front($exception->getMessage(), [
                    'fieldname' => $fieldName,
                    'type' => $type,
                ]);

                return;
            }

            if (! array_key_exists('target', $fieldOptions) || ! is_array($fieldOptions['target']) || empty($fieldOptions['target'])) {
                return;
            }
            $type = $fieldOptions['target'][0];
        }

        if (in_array(array_key_first($filters), ['uuid:in', 'uuid:eq']) && empty(Arr::first($filters))) {
            $query->whereDoesntHave(Str::camel($fieldName));

            return;
        }

        // NOTE: don't eager load!! it will conflict with permission scope mechanism
        //$query->with(Str::camel($field));
        $query->whereHas(
            Str::camel($fieldName),
            function ($query) use ($type, $filters) {
                $this->applyFilters($query, $type, $filters);
            }
        );
    }

    /**
     * Apply query filters
     */
    protected function applyFilters(\Illuminate\Database\Eloquent\Builder $query, string $type, ?array $customFilters = null): void
    {
        $filters = $customFilters ?? $this->filters;

        $table = type_to_table($type);

        if (! $table) {
            return;
        }

        foreach ($filters as $field => $value) {
            $sanitizedField = explode(':', $field);

            if ($type == 'users' && $sanitizedField[0] == 'langcode') {
                continue;
            }

            switch ($sanitizedField[0]) {
                case 'limit':
                case 'offset':
                case 'sort':
                case 'paginate':
                    // Do it on Illuminate\Database\Eloquent\Collection object.
                    break;
                case 'content_type':
                    // done later
                    break;
                case 'parent_uuid':
                    $query->join(
                        (string) DB::raw('`inside_pivots` AS `parent_pivots`'),
                        $table.'.uuid',
                        '=',
                        'parent_pivots.related_uuid'
                    )
                      ->where('parent_pivots.parent_uuid', '=', $value);
                    break;
                case 'archived':
                    if (! Package::has('inside-archive')) {
                        warn_front('Package archive is not installed');
                        continue 2;
                    }
                    try {
                        $query->leftJoin(
                            'inside_archives',
                            'inside_archives.uuid',
                            '=',
                            $table.'.uuid'
                        );
                        $query->where(
                            function ($subQuery) use ($value) {
                                if ($value) {
                                    $subQuery->whereHas(
                                        'archive',
                                        function ($archiveQuery) {
                                            $archiveQuery->where('date', '<=', date('Y-m-d H:i:s'));
                                        }
                                    );
                                } else {
                                    $subQuery->orwhereHas(
                                        'archive',
                                        function ($archiveQuery) {
                                            $archiveQuery->where('date', '>=', date('Y-m-d H:i:s'));
                                        }
                                    );
                                    $subQuery->orWhereDoesntHave('archive');
                                }
                            }
                        );
                    } catch (Exception $exception) {
                        warn_front('[ContentQuery::applyFilters] failed to apply archived filter -> '.
                          $exception->getMessage());
                        Log::error('[ContentQuery::applyFilters] failed to apply archived filter -> '.
                          $exception->getMessage());
                    }
                    break;
                case 'workflow':
                    if (! Package::has('inside-workflow')) {
                        warn_front('Package workflow is not installed');
                        continue 2;
                    }
                    try {
                        $query->whereHas(
                            'workflow',
                            function ($workflowQuery) use ($value) {
                                $workflowQuery->where('status', '=', $value);
                            }
                        );
                    } catch (Exception $exception) {
                        warn_front('[ContentQuery::applyFilters] failed to apply workflow filter -> '.
                          $exception->getMessage());
                        Log::error('[ContentQuery::applyFilters] failed to apply workflow filter -> '.
                          $exception->getMessage());
                    }
                    break;
                case 'slug':
                    $modelOptions = InsideSchema::getModelOptions($type);
                    $sluggable = (bool) ($modelOptions['aliasable'] ?? false);
                    if (! $sluggable) {
                        warn_front('[ContentQuery::applyFilters] asking slug on an non sluggable type ['.$type.']');
                        break;
                    }
                    // TODO : Ugly, for translation purpose.
                    if (! is_array($value)) {
                        $value = explode('/', $value);
                    }

                    $uuids = DB::table($table)
                      ->select('inside_slugs.uuid')
                      ->leftjoin('inside_slugs', 'inside_slugs.uuid', $table.'.uuid')
                      ->where('uuid_host', fn ($query) => $query
                        ->select('uuid_host')
                        ->from('inside_slugs')
                        ->leftjoin($table, 'inside_slugs.uuid', $table.'.uuid')
                        ->whereIn('slug', $value)
                        ->limit(1)
                      )->pluck('uuid')->toArray();

                    $query->join('inside_slugs', $table.'.uuid', '=', 'inside_slugs.uuid')->when(
                        ! isset($sanitizedField[1]) || $sanitizedField[1] != 'ne',
                        function ($query) use ($uuids) {
                            return $query->whereIn(
                                'inside_slugs.uuid',
                                $uuids
                            );
                        },
                        function ($query) use ($uuids) {
                            return $query->whereNotIn(
                                'inside_slugs.uuid',
                                $uuids
                            );
                        }
                    );

                    break;
                case '$or':
                    // '$or': [{ filter1 }, { filter2 }]
                    if (! is_array($value)) {
                        warn_front('[ContentQuery::applyFilters] $or needs an array');
                        break;
                    }

                    $query->where(
                        function ($subquery) use ($value, $type) {
                            foreach ($value as $filterFieldName => $subFilters) {
                                if (is_array($value) && ! is_numeric($filterFieldName)) {
                                    $subFilters = [$filterFieldName => $subFilters];
                                }
                                $subquery->orWhere(
                                    function ($filterQuery) use ($subFilters, $type) {
                                        $this->applyFilters($filterQuery, $type, $subFilters);
                                    }
                                );
                            }
                        }
                    );
                    break;
                case 'date_offset':
                    $this->filterByDateOffsetQuery($value, $query);
                    break;
                case 'date_range':
                    $this->filterByDateRangeQuery($type, $value, $query);
                    break;
                case 'date_sort':
                    $this->sortByDate($value, $query);
                    break;
                default:
                    $filter = explode(':', $field);

                    if (! Str::endsWith($type, '_menus') && ! InsideSchema::isSystemField($filter[0])
                      && ! InsideSchema::hasField($type, $filter[0])
                      && ! is_array($value)
                    ) {
                        warn_front('[ContentQuery::applyFilters] field '.$filter[0].' is neither a field of '.$type.' nor value is an array');
                        continue 2;
                    }

                    $operator = isset($filter[1]) ? ContentHelper::getOperatorFromString($filter[1]) : '=';
                    if ($operator == 'not in') {
                        if (! is_array($value[0])) {
                            $query->whereNotIn($table.'.'.$filter[0], $value);
                            break;
                        }

                        // "news_categories:notin": [{"hide":true}]
                        $subquery = DB::table($table)
                          ->join('inside_pivots', $table.'.uuid', '=', 'inside_pivots.parent_uuid')
                          ->join('inside_content_'.$filter[0], 'inside_pivots.related_uuid', '=', 'inside_content_'.$filter[0].'.uuid')
                          ->select($table.'.uuid');
                        collect($value[0])->each(function ($value, $column) use ($subquery, $filter) {
                            $subquery->orWhere(function ($query) use ($column, $value, $filter) {
                                $query->where('inside_content_'.$filter[0].'.'.$column, '=', $value);
                            });
                        });
                        $query->whereNotIn($table.'.uuid', $subquery->get()->pluck('uuid')->toArray());
                        break;
                    }

                    if ($operator === 'in') {
                        $field = $table.'.'.$filter[0];
                        if (in_array($filter[0], ['uuid', 'uuid_host'])) {
                            $query->whereIn($field, $value);
                            if (! isset($this->filters['sort'])) {
                                $query->orderByField($field, $value);
                            }
                        } else {
                            $query->whereIn($field, $value);
                        }

                        break;
                    }

                    if ($operator == 'null') {
                        $query->whereNull($table.'.'.$filter[0]);
                        break;
                    }

                    if ($operator == 'notnull') {
                        $query->whereNotNull($table.'.'.$filter[0]);
                        break;
                    }

                    if (is_array($value)) {
                        $this->applyRelation($query, $filter[0], $value, $type);
                        break;
                    }

                    // Manage special value now() for date comparaison
                    if (Str::lower($value) == 'now()') {
                        $value = now();
                    }

                    $query->where($table.'.'.$filter[0], $operator, $value);
                    break;
            }
        }
    }

    /**
     * Prepare a query
     *
     * @param  string  $type
     * @param  Request  $request
     */
    public function prepareQuery(string $type, Request $request): void
    {
        $this->extractFieldsInput($request);
        $this->extractFiltersInput($type, $request);

        $this->types = array_filter(
            explode(',', $type),
            function ($type) {
                return InsideSchema::hasContentType($type);
            }
        );
    }

    /**
     * Get query depending on filters & fields
     *
     * @return Builder
     * @throws Exception
     */
    public function getQuery(): Builder
    {
        if (count($this->types) > 1) {
            warn_front('[ContentQuery::getQuery] may only work with one type', ['types' => $this->types]);
            throw new Exception('[ContentQuery::getQuery] may only work with one type');
        }
        $type = Arr::first($this->types);
        if (! $type) {
            warn_front('[ContentQuery::getQuery] need to choose one type', ['types' => $this->types]);
            throw new Exception('[ContentQuery::getQuery] need to choose one type');
        }

        if (! InsideSchema::hasContentType($type)) {
            warn_front('[ContentQuery::getQuery] type ['.$type.'] is not a valid content type', ['types' => $this->types]);
            throw new Exception('[ContentQuery::getQuery] type ['.$type.'] is not a valid content type');
        }

        $query = call_user_func(type_to_class($type).'::'.$this->queryCallback);

        // Apply filters
        $this->applyFilters($query, $type);

        // Only select main columns
        $query->select(type_to_table($type).'.*');

        // limited result without pagination
        if (! isset($this->filters['paginate'])) {
            if (isset($this->filters['limit'])) {
                $query->limit($this->filters['limit']);
            }

            if (isset($this->filters['offset'])) {
                $query->offset($this->filters['offset']);
            }
        }

        // Order
        if (isset($this->filters['sort'])) {
            if ($this->filters['sort'] == 'random') {
                $query->inRandomOrder();
            } else {
                if (! is_array($this->filters['sort'])) {
                    $this->filters['sort'] = [$this->filters['sort']];
                }

                foreach ($this->filters['sort'] as $sort) {
                    $filter = explode(':', $sort);
                    $direction = $filter[1] ?? 'asc';
                    $query->orderBy(type_to_table($type).'.'.$filter[0], $direction);
                }
            }
        }

        return $query;
    }

    /**
     * Execute a multiple types query
     */
    public function doMultipleTypesQuery(): void
    {
        foreach ($this->types as $type) {
            $table = type_to_table($type);

            if (! $table || ! Schema::hasTable($table)) {
                warn_front('[ContentQuery::doMultipleTypesQuery] type ['.$type.'] has no table', ['types' => $this->types]);
                continue;
            }

            $query =
              call_user_func('\Inside\Content\Models\Contents\\'.Str::studly($type).'::'.$this->queryCallback);

            $this->applyFilters($query, $type);
            if (in_array($type, $this->userContentSeenTypes['types'])) {
                $contents = $this->addUserHasReadContentAttribute($query, $type);
            } else {
                $contents = $query->addSelect(type_to_table($type).'.*')->get();
            }

            foreach ($contents as $content) {
                $this->collection->push($content);
            }
        }

        $this->applyCollectionSort();

        $this->applyCollectionPaginate();
    }

    /**
     * Execute a simple type query
     */
    public function doSimpleTypeQuery(): void
    {
        foreach ($this->types as $type) {
            $query = call_user_func(type_to_class($type).'::'.$this->queryCallback);
            $table = type_to_table($type);
            $this->applyFilters($query, $type);

            if (! isset($this->filters['paginate'])) {
                if (isset($this->filters['limit'])) {
                    $query->limit($this->filters['limit']);
                }

                if (isset($this->filters['offset'])) {
                    $query->offset($this->filters['offset']);
                }
            }

            if (isset($this->filters['sort'])) {
                if ($this->filters['sort'] == 'random') {
                    $query->inRandomOrder();
                } elseif (! is_array($this->filters['sort']) && Str::startsWith($this->filters['sort'], 'field:')) {
                    $field = Str::after($this->filters['sort'], 'field:');
                    if (! array_key_exists($field.':in', $this->filters) || ! is_array($this->filters[$field.':in'])) {
                        throw new Exception('You must set the field "'.$field.':in" filter to use the "field" sort');
                    }

                    $query->orderByField($table.$field, $this->filters["$field:in"]);
                } else {
                    if (! is_array($this->filters['sort'])) {
                        $this->filters['sort'] = [$this->filters['sort']];
                    }

                    foreach ($this->filters['sort'] as $sort) {
                        $filter = explode(':', $sort);
                        $direction = $filter[1] ?? 'asc';

                        $sort = $table.'.'.$filter[0];

                        if ($filter[0] === 'archive.date') {
                            $sort = 'inside_archives.date';
                        }

                        $query->orderBy($sort, $direction);
                    }
                }
            }

            // Statistics fields
            $this->addStatisticsFieldsToQuery($type, $query);

            if (isset($this->filters['limit']) && isset($this->filters['paginate']) && $this->filters['paginate']) {
                $this->collection = $query->addSelect(type_to_table($type).'.*')->paginate($this->filters['limit']);

                return;
            }

            if (in_array($type, $this->userContentSeenTypes['types'])) {
                $this->collection = $this->addUserHasReadContentAttribute($query, $type);
            } else {
                $this->collection = $query->addSelect(type_to_table($type).'.*')->get();
            }
        }
    }

    /**
     * Do query
     *
     * @return Collection|LengthAwarePaginator|\Illuminate\Support\Collection
     */
    public function doQuery()
    {
        if (count($this->types) > 1) {
            $this->doMultipleTypesQuery();
        } else {
            $this->doSimpleTypeQuery();
        }

        return $this->collection;
    }

    /**
     * Transformed fetched contents
     *
     * @return array
     */
    public function transformCollection(): array
    {
        if (get_class($this->collection) === 'Illuminate\Pagination\LengthAwarePaginator') {
            /** @var Collection $collection */
            $collection = $this->collection->getCollection();

            $transformed = $this->transformer->transformCollection($collection, $this->fields);

            $result = $this->collection->toArray();

            $result['data'] = [];

            if (isset($transformed['data'])) {
                $result['data'] = $transformed['data'];
            }

            return $result;
        }

        /** @var Collection $collection */
        $collection = $this->collection;

        return $this->transformer->transformCollection($collection, $this->fields);
    }

    /**
     * Trigger statistic log event
     */
    public function logAnalytic(): void
    {
        $statisticTypes = config('statistics.types', []);
        if (empty(array_intersect(array_keys($statisticTypes), $this->types))) {
            return;
        }
        // We only log when it is asked for a specific content !
        $isOneContentQuery = false;
        foreach ($this->filters as $filter => $value) {
            $operator = 'eq';
            if (strpos($filter, ':')) {
                [$filter, $operator] = explode(':', $filter);
            }
            if ($operator === 'eq' && in_array($filter, ['slug', 'uuid']) && ! empty($value)) {
                $isOneContentQuery = true;
                break;
            }
        }
        if (! $isOneContentQuery) {
            return; // Don't log!
        }

        $items = $this->collection;
        if (get_class($this->collection) === 'Illuminate\Pagination\LengthAwarePaginator') {
            $items = $this->collection->items();
        }
        $user = Auth::user();
        foreach ($items as $item) {
            if (! array_key_exists(class_to_type($item), $statisticTypes)) {
                continue;
            }
            event(new AnalyticViewLogEvent(class_to_type($item), $item, $user, now()));
        }
    }

    /**
     * Check if path is authorized to bypass scope.
     */
    public function isAuthorized(): void
    {
        if (! isset($this->path) || $this->path == null) {
            return;
        }
        foreach ($this->authorizedPaths as $path) {
            if (str_contains($this->path, $path)) {
                $this->queryCallback = 'withoutGlobalScopes';
                break;
            }
        }
    }

    /**
     * Custom feature for desk to check if user already seen the news or not.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param string $type
     * @return Collection
     */
    private function addUserHasReadContentAttribute(\Illuminate\Database\Eloquent\Builder $query, string $type): Collection
    {
        $authUser = Auth::user();
        $userUuid = $authUser->uuid;
        $typeTableName = type_to_table($type);

        $hasRead = str('(SELECT COUNT(*) FROM inside_user_content_seen WHERE inside_user_content_seen.seen_uuid = ?.uuid_host AND inside_user_content_seen.user_uuid = ?) AS user_has_read')
          ->replaceArray('?', [$typeTableName, Str::wrap($userUuid, '\'')]);

        return $query->addSelect([$typeTableName.'.*', DB::raw($hasRead)])->get();
    }
}
