<?php

namespace Inside\Content\Services;

use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Auth;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Contents\ConnexionPopups;
use Inside\Reaction\Facades\Reactions;
use Inside\Reaction\Models\Reaction;
use Twilio\Rest\Events\V1\ReadEventTypeOptions;

/**
 * Class PopinService
 */
class PopinService
{
    private const EVERY_CONNECTION = 'every_connection';

    private const FIRST_CONNECTION = 'first_connection';

    private const ALLOWED_RECURRENCES = [
        self::EVERY_CONNECTION,
        self::FIRST_CONNECTION,
    ];

    public function getResponse(string $recurrence, string $langcode = ''): JsonResponse
    {
        $popin = null;
        if (in_array($recurrence, self::ALLOWED_RECURRENCES)) {
            $popin = $this->getLastValidPopin($recurrence, $langcode);
        }

        return response()->json([
            'popin' => $popin,
        ]);
    }

    public function cleanPastReactions(ConnexionPopups $popin): void
    {
        /** @var User $user */
        $user = Auth::user();
        $today = Carbon::today();

        Reaction::query()
            ->where('type', 'popin')
            ->where('user_uuid', $user->uuid)
            ->where('reactionable_uuid', $popin->uuid)
            ->whereDate('created_at', '<=', $today)
            ->orderBy('created_at', 'DESC')
            ->each(function (Reaction $reaction) {
                Reactions::delete('popin', $reaction->reactionable_uuid, $reaction->user_uuid, $reaction->langcode);
            });
    }

    private function getLastValidPopin(string $recurrence, string $langcode = ''): ?ConnexionPopups
    {
        $popin = $this->getLastPopin($langcode);

        if (! $popin instanceof ConnexionPopups) {
            return null;
        }

        /** @var User $user */
        $user = Auth::user();

        $lastReaction = Reaction::query()
            ->where('type', 'popin')
            ->where('user_uuid', $user->uuid)
            ->where('reactionable_uuid', $popin->uuid)
            ->latest()
            ->first();

        if (is_null($lastReaction)) {
            return $popin;
        }

        $lastReactionDate = get_date($lastReaction->updated_at);

        return $recurrence === self::FIRST_CONNECTION ?
            null :
            (($user->last_login_at > $lastReactionDate) ? $popin : null);
    }

    private function getLastPopin(string $langcode = ''): ?ConnexionPopups
    {
        return ConnexionPopups::query()
            ->where('status', 1)
            ->when($langcode, function ($query) use ($langcode) {
                $query->where('langcode', $langcode);
            })
            ->orderBy('updated_at', 'DESC')
            ->first();
    }
}
