<?php

namespace Inside\Content\Services;

use Illuminate\Http\JsonResponse;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;
use Inside\Content\Models\Content;
use Inside\Facades\Http;

class OpenWeatherService
{
    private string $city;

    private string $country;

    private Content $content;

    public function __construct(private string $apiKey, private string $apiUrl)
    {
    }

    public function setContent(Content $content): self
    {
        $this->content = $content;

        return $this;
    }

    public function getWeather(): array
    {
        $this->discoverLocationFromContent();

        if (empty($this->city) && empty($this->country)) {
            return [];
        }

        if (empty($this->apiKey)) {
            return [];
        }

        $query = $this->buildQuery();

        return $this->fetchWeatherData($query);
    }

    private function discoverLocationFromContent(): void
    {
        $cityField = config('openweathermap.city_field');
        $countryField = config('openweathermap.country_field');
        $content = $this->content;

        $city = $content->{$cityField} ?? config('openweathermap.default_city');
        $country = $content->{$countryField} ?? config('openweathermap.default_country');

        $this->city = is_string($city) ? $city : ($city->first()?->title ?? config('openweathermap.default_city'));
        $this->country = is_string($country) ? $country : ($country->first()?->title ?? config('openweathermap.default_country'));
    }

    private function buildQuery(): string
    {
        $query = urlencode($this->city);

        if (! empty($this->country)) {
            $query = empty($query) ? urlencode($this->country) : $query.','.urlencode($this->country);
        }

        return $query;
    }

    private function fetchWeatherData(string $query): array
    {
        $cachedData = Cache::tags(['weather'])->get($query);

        if ($cachedData !== null) {
            return [
                $cachedData,
            ];
        }

        try {
            $response = Http::get(
                sprintf($this->apiUrl, $query, $this->apiKey)
            );

            if ($response->status() !== 200) {
                return [];
            }

            $weatherData = $response->json();
            Cache::tags(['weather'])->put($query, $weatherData, config('openweathermap.cache_duration'));
        } catch (\Exception $e) {
            $weatherData = [];
        }

        return [
            $weatherData,
        ];
    }

    public function getWeatherByCoord(float $lat, float $lon): array
    {
        if (empty($this->apiKey)) {
            return [];
        }

        try {
            $response = Http::get(
                sprintf('https://api.openweathermap.org/data/2.5/weather?units=metric&lat=%s&lon=%s&appid=%s', $lat, $lon, $this->apiKey)
            );

            if ($response->status() !== 200 || empty($response->json())) {
                return [];
            }

            $weatherData = $response->json();

            return $weatherData;
        } catch (\Exception $e) {
            return [];
        }
    }

    public function getLocationsForLatlong(float $lat, float $lon): array
    {
        if (empty($this->apiKey)) {
            return [];
        }

        try {
            $response = Http::get(
                sprintf('https://api.openweathermap.org/geo/1.0/reverse?lat=%s&lon=%s&appid=%s', $lat, $lon, $this->apiKey)
            );

            if ($response->status() !== 200 || empty($response->json())) {
                return [];
            }

            $locations = $response->json();

            return collect($locations)
                ->map(fn ($weather) => [
                    'value' => [$weather['lat'], $weather['lon']],
                    'label' => $weather['name'].', '.$weather['country'],
                ])
                ->values()
                ->toArray();
        } catch (\Exception $e) {
            return [];
        }
    }

    public function getLocationsForCity(string $city): array
    {
        if (empty($this->apiKey)) {
            return [];
        }

        $query = urlencode($city);

        try {
            $response = Http::get(
                sprintf('https://api.openweathermap.org/data/2.5/find?q=%s&limit=5&appid=%s', $query, $this->apiKey));

            if ($response->status() !== 200 || empty($response->json())) {
                return [];
            }

            $locations = $response->json();

            $weatherData = collect($locations['list'])
                ->map(fn ($weather) => [
                    'value' => [$weather['coord']['lat'], $weather['coord']['lon']],
                    'label' => $weather['name'].', '.$weather['sys']['country'],
                ])
                ->values()
                ->toArray();

            return $weatherData;
        } catch (\Exception $e) {
            return [];
        }
    }
}
