<?php

declare(strict_types=1);

namespace Inside\Content\Services\Managers;

use Exception;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Inside\Content\Events\ContentSynchedEvent;
use Inside\Content\Events\ProcessingUpdate;
use Inside\Content\Models\Content;
use Inside\Content\Models\Field;
use Inside\Content\Models\Model;
use Inside\Content\Models\Section;
use Ramsey\Uuid\Uuid;

/**
 * Base entity manager service.
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class BaseEntityManager
{
    protected string $class;

    public array $data;

    protected ?string $domain = null;

    protected array $fields = [];

    protected array $references = [];

    protected array $originalReferences = [];

    /**
     * Attributes list
     *
     * @var array
     */
    protected const ATTRIBUTE_KEYS = [
        'uuid',
        'uuid_host',
        'status',
        'langcode',
        'title',
        'author',
        'author_id',
        'update_author',
        'pid',
        'created_at',
        'updated_at',
        'published_at',
        'image',
    ];

    public function __construct(string $bundle, array $data)
    {
        if (isset($data['bundle']) && isset($data[$data['bundle']]) && ! array_key_exists('pid', $data)) {
            // Here, we have a model that handle pid, but it is not set in our data so let it be null
            $data['pid'] = null;

            if (! empty($data[$data['bundle']]) && ! is_array($data[$data['bundle']][0])) {
                $data['pid'] = $data[$data['bundle']][0];
            }
        }

        $this->class = $this->getModel($bundle);
        $this->data = $data;
        $this->fields = $this->getFields($this->class);
    }

    /**
     * Create content
     * @throws Exception
     */
    public function create(): void
    {
        if (isset($this->data['uuid'])) {
            $this->update();

            return;
        }

        // If for some reason published_at is missing, let's default to now
        if (! array_key_exists('published_at', $this->data) && $this->data['type'] !== 'menu_link_content' && class_to_type($this->class) !== 'comments') {
            $this->data['published_at'] = $this->data['created_at'] ?? now();
        }
        // From drupal author_id is missing, let's set it
        if (! array_key_exists('author_id', $this->data) && isset($this->data['author']) && $this->data['type'] !== 'menu_link_content') {
            $this->data['author_id'] = $this->data['author'];
        }
        // From drupal update_author is missing, let's set it
        if (! array_key_exists('update_author', $this->data) && isset($this->data['author'])) {
            $this->data['update_author'] = $this->data['author'];
        }

        $this->data['uuid'] = Uuid::uuid4()->toString();

        $filtered = array_intersect_key($this->data, array_flip(static::ATTRIBUTE_KEYS));
        /** @var Content|Section $content */
        $content = new $this->class();
        $content->forceFill($filtered);
        $this->process($content);
    }

    /**
     * Update content
     * @throws Exception
     */
    public function update(): void
    {
        if (! isset($this->data['uuid'])) {
            $this->create();
        }

        // From drupal author_id is missing, let's set it
        if (! array_key_exists('author_id', $this->data) && isset($this->data['author']) && $this->data['type'] !== 'menu_link_content') {
            $this->data['author_id'] = $this->data['author'];
        }

        $query = call_user_func($this->class.'::withoutGlobalScopes');
        /** @var Content|Section|null $content */
        $content = $query->find($this->data['uuid']);

        if (is_null($content)) {
            $filtered = array_intersect_key($this->data, array_flip(static::ATTRIBUTE_KEYS));
            $class = $this->class;
            /** @var Content|Section $content */
            $content = new $class($filtered);
            $this->process($content);

            return;
        }

        // Created should never be changed on updates ( we use published_at for that )
        if (array_key_exists('created_at', $this->data)) {
            unset($this->data['created_at']);
        }

        // Updated should not be set manually.
        // We added the type users candition because if we update only pivots of the user and we unset here the
        // "updated_at" the updatedEvent does not triggered and the premission will not be updated.
        if (array_key_exists('updated_at', $this->data) && class_to_type($this->class) !== 'users') {
            unset($this->data['updated_at']);
        }

        $this->process($content);
    }

    /**
     * Delete content
     */
    public function delete(): void
    {
        if (! isset($this->data['uuid'])) {
            return;
        }

        $content = call_user_func($this->class.'::find', $this->data['uuid']);

        if (! $content) {
            return;
        }

        $content->delete();
    }

    /**
     * @throws Exception
     */
    protected function process(Content|Section $content): void
    {
        try {
            foreach ($this->fields as $field) {
                $content = $this->attachField($content, $field);
            }

            $content = $this->attachAttributes($content);
            $content = $this->attachReferences($content);
            ProcessingUpdate::dispatch($content);
            $content->save();

            if (($content->content_type != 'users') && ! Str::endsWith($content->content_type, '_menus')) {
                $this->processTranslation($content);
            }
        } catch (Exception $exception) {
            Log::error('[BaseEntityManager::process] failed => '.$exception->getMessage());

            throw $exception;
        }
    }

    /**
     * Attach references fields
     *
     * @param mixed $content
     *
     * @return mixed
     */
    protected function attachReferences(mixed $content): mixed
    {
        foreach ($this->fields as $field) {
            if ($field['type'] == 'reference'
                && ! in_array($field['name'], ['author', 'author_id', 'update_author'])
                && isset($field['options']['target'])) {
                if (! isset($this->references[$field['name']])) {
                    if ($this->originalReferences[$field['name']]
                        && $this->originalReferences[$field['name']]->isNotEmpty()
                        && $content instanceof Content
                    ) {
                        // Only trigger if something changed !
                        ContentSynchedEvent::dispatch(
                            $content,
                            $field['name'],
                            $this->originalReferences[$field['name']],
                            new Collection()
                        );
                    }
                    continue;
                }

                $targets = (array) $field['options']['target'];
                $className = Str::camel($field['name']);

                // ##IMPORTANT NOTE## We are using direct use of relation instead of our, because our has some filter
                // That prevent from deleting every relation
                $relation = $content->belongsToMany(
                    '\Inside\Content\Models\Contents\\'.Str::studly($targets[0] ?? $className),
                    'inside_pivots',
                    'parent_uuid',
                    'related_uuid'
                )->wherePivot('related_field', $field['name']);
                $relation->sync($this->references[$field['name']]);
                $newValues = $relation->get();

                $originalValue = $this->originalReferences[$field['name']] ?? null;

                if ($newValues != $originalValue && $content instanceof Content) {
                    // Only trigger if something changed !
                    ContentSynchedEvent::dispatch(
                        $content,
                        $field['name'],
                        $originalValue,
                        $newValues
                    );
                }
            }
        }

        return $content;
    }

    /**
     * Attach attributes fields
     *
     * @param mixed $content
     *
     * @return mixed
     */
    protected function attachAttributes($content)
    {
        $filtered = array_intersect_key($this->data, array_flip(static::ATTRIBUTE_KEYS));

        foreach ($filtered as $attribute => $value) {
            $content->{$attribute} = $value;
        }

        return $content;
    }

    /**
     * Process translation
     *
     * @param mixed $content
     *
     * @return void
     */
    protected function processTranslation($content)
    {
        $query = call_user_func($this->class.'::query');
        $translations = $query->where('uuid_host', $content->uuid_host)
            ->where('langcode', '!=', $content->langcode)
            ->get();

        // We need to recopy non-translatable field to other translations
        foreach ($translations as $translation) {
            foreach ($this->fields as $field) {
                $options = $field['options'];
                $translatable = $options['translatable'];

                // IMPORTANT: we assume reference are always translatable
                // otherwise this will be messy inside side
                if ($translatable || ($field['type'] == 'reference')) {
                    continue; // It's translatable, don't change anything
                }

                $translation = $this->attachField($translation, $field);
            }
            ProcessingUpdate::dispatch($translation);
            $translation->save();
        }
    }

    protected function attachField(Content|Section $content, array $field): Content|Section
    {
        $fieldName = $field['name'];

        if (! array_key_exists($fieldName, $this->data)) {
            return $content;
        }

        switch ($field['type']) {
            case 'reference':
                $className = Str::camel($fieldName);
                $targets = isset($field['options']['target']) ? (array) $field['options']['target'] : [];

                if (in_array($field['name'], ['author', 'author_id', 'update_author'])) {
                    $content->{$fieldName} = $this->data[$fieldName];
                    break;
                }

                // Be sure it is an array !
                if (is_string($this->data[$fieldName])) {
                    $this->data[$fieldName] = [$this->data[$fieldName]];
                }

                if (! empty($this->data[$fieldName])) {
                    $uuids = array_unique(array_filter($this->data[$fieldName]));

                    foreach ($uuids as $weight => $uuid) {
                        if (is_string($uuid) && ! empty($uuid)) {
                            foreach ($targets as $target) {
                                $relation = call_user_func(
                                    '\Inside\Content\Models\Contents\\'.Str::studly($target).'::find',
                                    $uuid
                                );

                                if ($relation) {
                                    $relation = $relation->getTranslationIfExists($content->langcode);

                                    $uuid = $relation->uuid;

                                    $this->references[$field['name']][$uuid] = [
                                        'weight' => $weight,
                                        'parent_type' => get_class($content),
                                        'parent_langcode' => $content->langcode,
                                        'related_type' => get_class($relation),
                                        'related_langcode' => $relation->langcode,
                                        'related_field' => $field['name'],
                                    ];

                                    break;
                                }
                            }
                        }
                    }
                } else {
                    $this->references[$field['name']] = [];
                }

                if (method_exists($content, $className)) {
                    // Reset all relation for this  field
                    // ##IMPORTANT NOTE## We are using direct use of relation instead of our,
                    // because our has some filter
                    // That prevent from deleting every relation
                    $relation = $content->belongsToMany(
                        '\Inside\Content\Models\Contents\\'.Str::studly($targets[0] ?? $className),
                        'inside_pivots',
                        'parent_uuid',
                        'related_uuid'
                    )->wherePivot('related_field', $field['name']);

                    $this->originalReferences[$field['name']] = $relation->get();
                    $relation->sync([]);
                }
                break;
            default:
                $content->{$fieldName} = $this->data[$fieldName];
                break;
        }

        return $content;
    }

    protected function getModel(string $bundle): string
    {
        return 'Inside\Content\Models\\'.$this->domain.'\\'.Str::studly($bundle);
    }

    /**
     * Return the fields of a specific model
     */
    protected function getFields(string $className): array
    {
        $fields = Field::whereHas(
            'model',
            function ($query) use ($className) {
                $query->whereClass($className);
            }
        )->whereNotIn('type', ['comment', 'section'])->get();

        if ($fields->isEmpty()) {
            return [];
        }

        return $fields->toArray();
    }
}
