<?php

namespace Inside\Content\Services;

use Illuminate\Contracts\Filesystem\Filesystem;
use Illuminate\Database\Query\Builder;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Storage;
use Inside\Content\Facades\Schema;
use Inside\Content\Models\Content;
use Inside\Content\Models\Contents\Documents;
use Inside\Content\Models\Contents\Folders;
use Inside\Content\Models\Contents\Tools;
use Inside\Content\Models\Contents\Users;
use Inside\Host\Bridge\BridgeContent;
use PharData;
use ZipArchive;

/**
 * Class GedDuplicationService
 */
class GedDuplicationService
{
    private Filesystem $disk;

    private string $prefix;

    private ?string $rootFolder;

    private array $stats = [
        'duplicated' => [

        ],
        'skiped' => [

        ],
        'exists' => [

        ],
    ];

    public function execute(): array
    {
        $this->disk = Storage::disk('local');

        $gedTree = $this->gedTree();
        $gedTree->each(function ($folder) {
            $this->duplicateFolder($folder);
        });

        return $this->stats;
    }

    public function setPrefix(string $prefix): self
    {
        $this->prefix = $prefix;

        return $this;
    }

    public function setRootFolder(?string $rootFolder): self
    {
        $this->rootFolder = $rootFolder;

        return $this;
    }

    private function gedTree(): Collection
    {
        /**
         * @var Builder $foldersQuery
         */
        $foldersQuery = type_to_class('folders')::query()
            ->where([
                'pid' => null,
                'status' => true,
            ]);

        if (! is_null($this->rootFolder)) {
            $foldersQuery->where('uuid', $this->rootFolder);
        }

        return $foldersQuery->get();
    }

    private function duplicateFolder(Folders $folder, ?string $parentFolderUuid = null): void
    {
        $data = $this->getData($folder, $parentFolderUuid);

        if (empty($data)) {
            $this->stats['skiped']['folder'][] = $folder->title;

            return;
        }

        $newFolderUuid = $this->insertOrUpdateContent('folders', $data);

        $this->stats['duplicated']['folder'][] = $folder->title;

        $subFolders = $folder->getChildrenIfExist();

        $subFolders->each(function ($subfolder) use ($newFolderUuid) {
            $this->duplicateFolder($subfolder, $newFolderUuid);
        });

        $reverseDocumentsTypes = config('duplication.reverseContents');

        collect($reverseDocumentsTypes)->each(
            fn ($reverseDocumentsType) =>  $this->duplicateDocuments(
                $folder,
                $reverseDocumentsType,
                $newFolderUuid
            )
        );
    }

    private function getData(Content|Folders $content, ?string $folderUuid): array
    {
        $data = array_except($content->getAttributes(), [
            'uuid', 'uuid_host', 'author', 'author_id', 'update_author', 'pid', 'created_at', 'updated_at', 'published_at',
        ]);

        // Do not duplicate, duplicated contents
        if (str($data['title'])->contains($this->prefix)) {
            return [];
        }

        $data['title'] = str($data['title'])->append(' '.$this->prefix)->toString();

        foreach (['video', 'image', 'file'] as $fileField) {
            /**
             * @var string $originalFilePath
             */
            $originalFilePath = $content->{$fileField};
            if (! empty($originalFilePath) && $this->disk->exists($originalFilePath)) {
                $data[$fileField] = $this->duplicateFile($originalFilePath);
            }
        }

        if ($folderUuid) {
            $data['folders'] = $folderUuid;
        }

        return $data;
    }

    private function duplicateDocuments(
        Folders $folder,
        string $reverseDocumentsType,
        string $newFolderUuid
    ): void {
        $documents = $folder->{$reverseDocumentsType}();
        if ($documents->count() < 1) {
            return;
        }

        $documents->each(function ($document) use ($newFolderUuid, $reverseDocumentsType) {
            $contentType = str($reverseDocumentsType)->after('reverse')->snake()->toString();
            $data = $this->getData($document, $newFolderUuid);
            if (empty($data)) {
                $this->stats['skiped'][$contentType][] = $document->title;

                return;
            }

            $sections = $this->getSections($document);
            if (! empty($sections)) {
                $data['content'] = $sections;
            }

            $newDocument = $this->insertOrUpdateContent($contentType, $data);
            $this->stats['duplicated'][$contentType][] = $newDocument;
        });
    }

    private function getSections(Content $content): array
    {
        $sections = $content->getSectionContentAttribute();
        if ($sections->isEmpty()) {
            return [];
        }

        return collect(get_content_sections($content))->map(function ($section, $id) {
            $return = $section;
            $return['uuid'] = "_freshUUID_$id";
            $return['uuid_host'] = "_freshUUIDHOST_$id";
            foreach (['video', 'image', 'file'] as $fileField) {
                $originalFilePath = $return[$fileField];
                if (! empty($originalFilePath) && $this->disk->exists($originalFilePath)) {
                    $return[$fileField] = $this->duplicateFile($originalFilePath);
                }
            }

            return $return;
        })->toArray();
    }

    private function duplicateFile(string $originalFilePath): ?string
    {
        $isSuccess = false;
        $copyFile = '';
        if (! empty($originalFilePath) && $this->disk->exists($originalFilePath)) {
            $isSuccess = true;
            $copyFile = sprintf(
                '%s/%s',
                dirname($originalFilePath),
                str(basename($originalFilePath))->prepend('1-')->toString()
            );
            if (! $this->disk->exists($copyFile)) {
                $isSuccess = $this->disk->copy($originalFilePath, $copyFile);
            }
        }

        return $isSuccess ? $copyFile : null;
    }

    private function insertOrUpdateContent(string $contentType, array $data): ?string
    {
        $exists = type_to_class($contentType)::where('title', $data['title'])->get();

        if ($exists->isNotEmpty()) {
            $this->stats['exists'][$contentType][] = $exists->first()->uuid;

            return $exists->first()->uuid;
        }
        $bridge = new BridgeContent();

        return $bridge->contentInsert(
            type: $contentType,
            data: $data,
            creation: true,
            fromCli: true
        );
    }
}
