<?php

declare(strict_types=1);

namespace Inside\Content\Services\Forms;

use Carbon\Carbon;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Inside\Authentication\Models\User;
use Inside\Content\Contracts\ModelForm as ModelFormContract;
use Inside\Content\Contracts\RevisionService;
use Inside\Content\Contracts\SchemaService;
use Inside\Content\Exceptions\FieldSchemaNotFoundException;
use Inside\Content\Exceptions\ModelSchemaNotFoundException;
use Inside\Content\Models\Content;
use Inside\Content\Models\Section;
use Inside\Permission\Exodus\Actions\SortPrivileges\SortByDepth;
use Inside\Permission\Exodus\Enums\CapabilityEnum;
use Inside\Permission\Facades\Permission;
use stdClass;
use Symfony\Component\HttpFoundation\File\Exception\FileNotFoundException;
use Symfony\Component\HttpFoundation\File\MimeType\MimeTypeGuesser;

/**
 * Inside content form builder.
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class ModelForm implements ModelFormContract
{
    protected ?User $user = null;

    public function __construct(
        protected SchemaService $schemaService,
        protected RevisionService $revisionService
    ) {
        /** @var ?User $user */
        $user = Auth::user();
        $this->user = $user;
    }

    public function __invoke(string $type, string $id = null, bool $isDuplicate = false): ?array
    {
        if (! $this->schemaService->hasModel($type)) {
            Log::error('Error can\'t get Model from type ['.$type.']');

            return null;
        }
        $content = null;
        $langCode = null;
        $class = $this->schemaService->isSectionType($type) ? section_type_to_class($type) : type_to_class($type);
        if (! is_null($id)) {
            $content = call_user_func($class.'::find', $id);

            if ($content) {
                $langCode = $content->langcode;
            }
        }

        $form = $this->formatForm($type, $this->schemaService->getDisplayedFieldListing($type), $content, $isDuplicate);
        $isRevisionable = ! is_null($content)
            && $this->revisionService->isEnabled($content)
            && $content->revisions()->count() > 1;

        return [
            'form' => $form,
            'langcode' => $langCode,
            'metadata' => [
                'revisionable' => $isRevisionable,
            ],
        ];
    }

    public function formatForm(string $type, array $fields = null, Content | Section | null $content = null, bool $isDuplicate = false): ?array
    {
        $form = [];

        if (! $fields) {
            return null;
        }

        foreach ($fields as $key => $fieldName) {
            if (
                ($content !== null && $content->exists && $fieldName === 'password')
                || (in_array($fieldName, ['created_at', 'updated_at', 'published_at']))
            ) {
                unset($fields[$key]);
                continue;
            }

            // TODO: fix author wrong field name
            try {
                $fieldOptions = $this->schemaService->getFieldOptions($type, $fieldName);
            } catch (FieldSchemaNotFoundException) {
                unset($fields[$key]);
                continue;
            }

            // Get original field name
            $fieldOptions['name'] = $fieldName = $this->schemaService->getOriginalFieldName($fieldName);

            // Manage front config
            if (! isset($fieldOptions['front_field_config']) || $fieldOptions['front_field_config'] == '') {
                $fieldOptions['front_field_config'] = new stdClass();
            } else {
                $fieldOptions['front_field_config'] =
                    json_decode($fieldOptions['front_field_config'], true) ?? new stdClass();
            }

            $fieldOptions = $this->attachValue($fieldName, $fieldOptions, $content, $isDuplicate);
            $fieldOptions = $this->alterField($fieldOptions);

            if (! empty($fieldOptions['group'])) {
                $form = $this->addFieldGroup($fieldName, $form, $fieldOptions);
                continue;
            }
            $fieldOptions = $this->schemaService->formatFieldToLegacyFront($fieldName, $fieldOptions);
            $fieldOptions['options'] = $fieldOptions; // TODO: TO BE REMOVED FOR LEGACY
            $form[] = $fieldOptions;
        }

        // Add published_at system field
        if ($this->schemaService->isContentType($type) && $type !== 'users') {
            foreach ([
                'group_options',
                'group_advanced_options',
            ] as $group) {
                if (! isset($form[$group]['fields'])) {
                    continue;
                }
                $lastOnOffWidget = 0;
                $atLeastOne = false;
                foreach ($form[$group]['fields'] as $key => $field) {
                    if (Str::lower($field['widget']) != 'onoff') {
                        if ($atLeastOne) {
                            break;
                        } else {
                            $atLeastOne = true;
                        }
                    }
                    $lastOnOffWidget = $key;
                }
                if ($lastOnOffWidget > 0 && isset($form['group_options']['fields'])) {
                    // Only show publishable if we have at least status ( first onoff widget )
                    array_splice($form['group_options']['fields'], $lastOnOffWidget + 1, 0, [
                        $this->getPublishedAtFieldOptions($type, $content, $isDuplicate),
                    ]);
                } else {
                    // Simply add it
                    $form[$group]['fields'][] = $this->getPublishedAtFieldOptions($type, $content, $isDuplicate);
                }
                break;
            }
        }

        usort(
            $form,
            function ($a, $b) {
                return $a['weight'] <=> $b['weight'];
            }
        );

        return $form;
    }

    public function alterField(array $fieldOptions): array
    {
        $langCodes = array_keys($fieldOptions['title']);

        if ($fieldOptions['type'] == 'reference' && $fieldOptions['widget'] != 'autocomplete') {
            foreach ($fieldOptions['target'] as $target) {
                foreach ($langCodes as $langCode) {
                    $order = 'title';

                    if ($target == 'users') {
                        $order = 'lastname';
                    }
                    if (Permission::isSystemV2Enabled()) {
                        /** @var Content $class */
                        $class = type_to_class($target);
                        $user = auth()->user();
                        $query = $class::query();

                        $option = $class::query()
                            ->when($target !== 'users', fn ($query) => $query->where('langcode', $langCode))
                            ->where('status', true)
                            ->where(fn ($query) => $query
                                ->whereIn('pid', $query->select('uuid')->where('status', true))
                                ->orWhereNull('pid')
                            )
                            ->when(! $user->isSuperAdmin() && $class::isCategorizable(), fn ($query) => $query
                                ->whereIn('uuid_host', $user
                                    ->getAuthorizedCategorizableContentPrivileges()
                                    ->map->toArray()
                                    ->where('capability.name', CapabilityEnum::ASSIGN)
                                    ->pluck('index.uuid_host')
                                )
                            )
                            ->orderBy($order)
                            ->get()
                            ->mapWithKeys(function ($content) {
                                if (class_to_type($content) == 'users') {
                                    $content->title = $content->name;
                                }

                                return [$content->uuid => $content->only('uuid', 'title', 'pid')];
                            });

                        $fieldOptions['allowed_values'][$target][$langCode] = (new SortByDepth())->execute($option, 'pid', true)->values();
                    } else {
                        $fieldOptions['allowed_values'][$target][$langCode] = $this->getNested(
                            call_user_func(type_to_class($target).'::query')
                                ->when($target !== 'users', fn ($query) => $query->where('langcode', $langCode))
                                ->where(fn ($query) => $query->where('pid', '')->orWhereNull('pid'))
                                ->where('status', true)
                                ->orderBy($order)
                                ->get(),
                            $fieldOptions['classifiable'] ?? '',
                            $order
                        );
                    }
                }
            }

            // Handle default values
            if (isset($fieldOptions['default']) && ! empty($fieldOptions['default'])) {
                /** @var User $user */
                $user = Auth::user();
                $langCode = $user->langcode;

                foreach ($fieldOptions['default'] as &$option) {
                    $reference = call_user_func(type_to_class($fieldOptions['target'][0]).'::query')
                        ->where('uuid_host', $option)
                        ->where('langcode', $langCode)
                        ->where('status', true)
                        ->first();

                    if ($reference) {
                        $option = $reference->uuid;
                    }
                }
            }
        }

        if ($fieldOptions['type'] == 'section') {
            foreach ($fieldOptions['target'] as $target) {
                /** @var array $sectionForm */
                $sectionForm = $this($target);
                $fieldOptions['target_form'][$target] = $sectionForm['form'];
                $fieldOptions['options'] = $fieldOptions; // TODO: TO BE REMOVED FOR LEGACY
            }
        }

        if ($fieldOptions['type'] == 'timestamp' && $fieldOptions['default'] === null) {
            $fieldOptions['default'] =
                (! isset($fieldOptions['required']) || $fieldOptions['required'])
                    ? Carbon::now('UTC')->timestamp :
                    null;
        }

        return $fieldOptions;
    }

    public function addFieldGroup(string $fieldName, array $form, array $fieldOptions): array
    {
        $group = $fieldOptions['group']['id'];

        if (! isset($form[$group])) {
            $form[$group] = $fieldOptions['group'];
        }

        unset($fieldOptions['group']);

        $fieldOptions = $this->schemaService->formatFieldToLegacyFront($fieldName, $fieldOptions);
        $fieldOptions['options'] = $fieldOptions; // TODO: TO BE REMOVED ONLY FOR LEGACY
        $form[$group]['fields'][] = $fieldOptions;

        usort(
            $form[$group]['fields'],
            function ($a, $b) {
                return $a['weight'] <=> $b['weight'];
            }
        );

        return $form;
    }

    /**
     * Get nested construction of a collection
     *
     * Example to construct categories
     *
     *  Cat1
     *     SubCat1-1
     *     SubCat1-2
     *  Cat2
     *     SubCat2-1
     * ...
     */
    protected function getNested(Collection $values, string $qualifier = '', string $order = 'title'): array
    {
        $data = [];

        $addList = Permission::getAllowedCreationTypesForUser($this->user);

        foreach ($values as $element) {
            if (
                ! in_array(class_to_type($element), $addList)
                && ! $this->user?->can('create', $element)
                && ! $this->user?->can('view', $element)
                && ! $this->user?->isSuperAdmin()
            ) {
                continue;
            }

            $qualifierInfos = [];
            if (! empty($qualifier)) {
                $qualifier = Str::camel($qualifier);
                if ($element->{$qualifier} instanceof Collection) {
                    $qualifiers = $element->{$qualifier}->transform(
                        function ($qualifier) {
                            return [
                                'title' => $qualifier->title ?? 'N/A',
                                'color' => $qualifier->color ?? '#aaaaaa',
                            ];
                        }
                    );
                } else {
                    $qualifiers = [
                        [
                            'title' => $element->{$qualifier}->title ?? 'N/A',
                            'color' => $element->{$qualifier}->color ?? '#ffffff',
                        ],
                    ];
                }
                $qualifierInfos = [
                    'qualifiers' => $qualifiers,
                ];
            }

            $children = $element->belongsTo(get_class($element), 'uuid', 'pid')->orderBy($order)->get();
            if ($children->count() > 0) {
                $children = ['children' => $this->getNested($children, $qualifier)];
            } else {
                $children = [];
            }

            $data[] = [
                'title' => class_to_type($element) == 'users' ? $element->name : $element->title,
                'uuid' => $element->uuid,
            ] + $qualifierInfos + $children;
        }

        return $data;
    }

    public function attachValue(string $fieldName, array $field, Content | Section | null $content, bool $isDuplicated = false): array
    {
        if (is_null($content)) {
            return $field;
        }

        $value = $content->{$fieldName};

        if ($field['type'] == 'reference' && ! is_string($value)) {
            $values = $content->{Str::camel($fieldName)};

            $field['value'] = $values ? $values->pluck('uuid') : null;

            if ($values !== null) {
                foreach ($values as $value) {
                    $query = call_user_func(get_class($value).'::query');
                    $translations = $query->where('uuid_host', $value->uuid_host)
                        ->where('langcode', '!=', $content->langcode)->get(); // @phpstan-ignore-line

                    foreach ($translations as $translation) {
                        $field['translatableValues'][$value->uuid][$translation->langcode] = [
                            'title' => $translation->title,
                            'uuid' => $translation->uuid,
                        ];
                    }
                }
            }

            return $field;
        }

        if ($field['type'] == 'section') {
            return $this->attachSectionValue($fieldName, $content, $field, $isDuplicated);
        }

        $field['value'] = $content->{$fieldName};

        if ($isDuplicated) {
            $field['value'] = match ($fieldName) {
                'author' => $this->user?->uuid,
                'status' => true,
                default => $field['value'],
            };
        }

        //  By design, null timestamp should be Now !
        if ($field['type'] == 'timestamp' && $field['value'] === null) {
            // Note: design changes : null date should be null date! if field is not required
            $field['value'] = (! isset($field['required']) || $field['required']) ? Carbon::now()->timestamp : null;
        } elseif (in_array($field['type'], ['image', 'file']) && $field['value']) {
            try {
                $field['preview'] =
                    protected_file_url($content, $fieldName, true, $field['type'] == 'image' ? 'medium' : null);
                $realpath = Storage::disk('local')->path($field['value']);
                MimeTypeGuesser::reset();
                $guesser = MimeTypeGuesser::getInstance();
                $field['mimetype'] = $guesser->guess($realpath);
                $field['size'] = filesize($realpath);

                if ($field['type'] == 'image' && $field['mimetype'] !== 'image/svg+xml') {
                    [$width, $height] = getimagesize($realpath) ?: [null, null];

                    if ($width && $height) {
                        $field['width'] = $width;
                        $field['height'] = $height;
                    }
                }
            } catch (FileNotFoundException) {
                $field['value'] = null;
                unset($field['preview']);
                unset($field['mimetype']);
                unset($field['size']);
                unset($field['width']);
                unset($field['height']);
            }
        } elseif ($field['type'] === 'link') {
            if (
                isset($field['value']) && ! inside_link_is_external($field['value'])
                && ! Str::startsWith(
                    $field['value'],
                    'mailto:'
                )
            ) {
                if (! Str::startsWith($field['value'], config('app.url'))) {
                    $field['value'] = config('app.url').'/'.$field['value'];
                }
            }
        }

        return $field;
    }

    /**
     * Attach the section (and subsections) values
     */
    protected function attachSectionValue(string $fieldName, Content | Section | null $content, array $field, bool $isDuplicated = false): array
    {
        $values = $content->{Str::snake('section_'.$fieldName)}->sortBy('weight');

        foreach ($values as $sectionPos => $value) {
            $data = $value->toArray();

            foreach ($this->schemaService->getFieldListing(class_to_type($value)) as $sectionFieldName) {
                $fieldOption = $this->schemaService->getFieldOptions(class_to_type($value), $sectionFieldName);
                switch ($fieldOption['type']) {
                    case 'image':
                        try {
                            $path = $value->{$sectionFieldName} ?? null;
                            if ($path) {
                                $data['preview'] = protected_file_url($value, $sectionFieldName, true, 'medium');
                                $realpath = Storage::disk('local')->path($path);
                                MimeTypeGuesser::reset();
                                $guesser = MimeTypeGuesser::getInstance();
                                $data['mimetype'] = $guesser->guess($realpath);
                                $data['size'] = filesize($realpath);

                                [$width, $height] = getimagesize($realpath) ?: [null, null];

                                if ($width && $height) {
                                    $data['width'] = $width;
                                    $data['height'] = $height;
                                }
                            }
                        } catch (FileNotFoundException) {
                            $data[$sectionFieldName] = null;
                            unset($data['preview']);
                            unset($data['mimetype']);
                            unset($data['size']);
                            unset($data['width']);
                            unset($data['height']);
                        }
                        break;
                    case 'file':
                        try {
                            $path = $value->{$sectionFieldName} ?? null;
                            if ($path) {
                                $data['preview'] = protected_file_url(
                                    $value,
                                    $sectionFieldName,
                                    true
                                );
                                $realpath = Storage::disk('local')->path($path);
                                MimeTypeGuesser::reset();
                                $guesser = MimeTypeGuesser::getInstance();
                                $data['mimetype'] = $guesser->guess($realpath);
                                $data['size'] = filesize($realpath);
                            }
                        } catch (FileNotFoundException) {
                            $data[$sectionFieldName] = null;
                            unset($data['preview']);
                            unset($data['mimetype']);
                            unset($data['size']);
                            unset($data['width']);
                            unset($data['height']);
                        }
                        break;
                    case 'link':
                        $data['link'] = $value->{$sectionFieldName} ?? null;
                        if (
                            isset($data['link']) && ! inside_link_is_external($data['link'])
                            && ! Str::startsWith($data['link'], 'mailto:')
                        ) {
                            if (! Str::startsWith($data['link'], config('app.url'))) {
                                $data['link'] = config('app.url').'/'.$data['link'];
                            }
                        }
                        break;
                    case 'reference':
                        $uuids = DB::table('inside_pivots')->where('parent_uuid', $value->uuid)
                            ->pluck('related_uuid');
                        $class = get_class($value);
                        if (! is_string($class)) {
                            break;
                        }
                        $reference = get_reference_class($class, $sectionFieldName);
                        if (! is_string($reference)) {
                            break;
                        }

                        $cardinality = $fieldOption['cardinality'];
                        if ($cardinality === 1) {
                            $data[$sectionFieldName] = $uuids->first();
                        } else {
                            $data[$sectionFieldName] = $uuids;
                        }

                        break;
                    case 'section':
                        $subSectionFields = $this->schemaService->getFieldListing(class_to_type($value));

                        // Get first field
                        $subSectionFieldName = Arr::first($subSectionFields);
                        try {
                            $subSectionFieldOptions =
                                $this->schemaService->getFieldOptions(class_to_type($value), $subSectionFieldName);
                            $subSectionFieldOptions =
                                $this->schemaService->formatFieldToLegacyFront($subSectionFieldName, $subSectionFieldOptions);
                            $subSectionFieldOptions['options'] = $subSectionFieldOptions; // TODO: TO BE REMOVED FOR LEGACY
                            $data[$sectionFieldName] = $this->alterField(
                                $this->attachSectionValue($subSectionFieldName, $value, $subSectionFieldOptions, $isDuplicated)
                            );
                        } catch (FieldSchemaNotFoundException | ModelSchemaNotFoundException $exception) {
                            Log::error(
                                '[ModelForm::attachSectionValue] failed to find subsection => '.
                                $exception->getMessage()
                            );
                        }

                        // TODO: champ multiple dans la section de premier niveau ( non supporté par le front pour le moment )
                        /**
                         * $data[$sectionFieldName] = [];
                         * foreach ($subSectionFields as $subSectionFieldName) {
                         * try {
                         * $subSectionFieldOptions =
                         * $this->schemaService->getFieldOptions(class_to_type($value), $subSectionFieldName);
                         * $subSectionFieldOptions            = $this->schemaService->formatFieldToLegacyFront($subSectionFieldName, $subSectionFieldOptions);
                         * $subSectionFieldOptions['options'] = $subSectionFieldOptions;
                         * } catch (FieldSchemaNotFoundException | ModelSchemaNotFoundException $e) {
                         * continue;
                         * }
                         * $data[$sectionFieldName][] = $this->alterField($this->attachSectionValue($sectionFieldName, $value, $subSectionFieldOptions));
                         * }
                         */
                        break;
                }
            }

            $data = array_merge(
                [
                    'section_type' => $value->section_type,
                    'weight' => $value->weight,
                    'uuid' => $value->uuid,
                ],
                $data
            );

            if ($isDuplicated) {
                $data['uuid'] = '_freshUUID_'.$sectionPos;
                $data['uuid_host'] = '_freshUUIDHOST_'.$sectionPos;
            }

            $field['value'][] = $data;
        }

        return $field;
    }

    protected function getPublishedAtFieldOptions(string $type, Content | Section | null $content = null, bool $isDuplicated = false): array
    {
        $weight = $value = null;
        $titles = $descriptions = [];

        if ($this->schemaService->hasField($type, 'status')) {
            $createdFieldOptions = $this->schemaService->getFieldOptions($type, 'status');
            $weight = $createdFieldOptions['weight'];
        }

        // Prepare labels
        foreach (list_languages() as $locale) {
            $titles[$locale] = __('contents.fields.published_at.title', [], $locale);
            $descriptions[$locale] = __('contents.fields.published_at.description', [], $locale);
        }

        if (! $isDuplicated && ! is_null($content)) {
            $value = $content->published_at;
        }

        $options = [
            'name' => 'published_at',
            'required' => true,
            'translatable' => false,
            'cardinality' => 1,
            'weight' => $weight,
            'default' => now()->timestamp,
            'widget' => 'datetime',
            'title' => $titles,
            'description' => $descriptions,
            'value' => $value,
        ];
        // Silly front TODO: remove a day ...
        $options['options'] = $options; // TODO: TO BE REMOVED FOR LEGACY

        return $options;
    }
}
