<?php

namespace Inside\Content\Services\Exporters;

use Closure;
use Exception;
use Illuminate\Http\Request;
use Inside\Content\Facades\Schema;
use Inside\Content\Services\Queries\ContentQuery;
use Inside\Database\Eloquent\Builder;
use Maatwebsite\Excel\Concerns\Exportable;
use Maatwebsite\Excel\Concerns\FromQuery;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithColumnFormatting;
use Maatwebsite\Excel\Concerns\WithCustomCsvSettings;
use Maatwebsite\Excel\Concerns\WithDrawings;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Events\AfterSheet;
use Maatwebsite\Excel\Events\BeforeExport;

/**
 * Class ContentExport
 */
class ContentExport implements FromQuery, WithEvents, WithDrawings, WithMapping, WithColumnFormatting, WithHeadings, ShouldAutoSize, WithCustomCsvSettings
{
    use Exportable;
    use ContentExportable;

    protected array $input;

    protected string $type;

    protected array|string|null $columns = null;

    /**
     * @throws Exception
     */
    public function __construct(
        protected array $config
    ) {
        if (! is_array($config['input']) || ! is_string($config['content_type'])) {
            throw new Exception('[ContentExport] wrong config');
        }

        // Preparing query
        $this->input = $config['input'];
        $this->type = $config['content_type'];
    }

    /**
     * load Columns when needed
     */
    protected function loadColumns(): void
    {
        if (! is_null($this->columns)) {
            return;
        }

        if (isset($this->config['config']) && $this->config['config']) {
            $importerConfigs = config('imports');
            $this->columns = $importerConfigs[$this->config['config']]['default_exports'];
        } elseif ($this->config['fields']) {
            $this->columns = $this->config['fields'];
            if (is_string($this->columns)) {
                $this->columns = json_decode($this->columns, true) ?? [];
            }
        }
        // Check columns
        foreach ($this->columns as $key => $column) {
            if (is_array($column)) {
                $column = (string) array_key_first($column);
                if (! $this->isSystemColumn($column) && ! $this->isExportable($this->type, $column)) {
                    unset($this->columns[$key]);
                }
            }
        }
    }

    /**
     * Retrieve the query
     *
     * @throws Exception
     */
    public function query(): Builder
    {
        $service = new ContentQuery();
        $request = new Request();
        $request->merge($this->input);
        $service->prepareQuery($this->type, $request);

        $query = $service->getQuery();
        $uuids = $request->get('uuids', []);
        if (! empty($uuids)) {
            $query->whereIn(type_to_table($this->type).'.uuid', $uuids);
        }

        return $query;
    }

    /**
     * mapping column
     *
     * @throws Exception
     */
    public function map(mixed $information): array
    {
        $this->loadColumns();
        $mapped = [];
        if (is_array($this->columns)) {
            $this->addColumns($information, $this->columns, $mapped, $this->type);
            ksort($mapped);
        }

        return array_values($mapped);
    }

    /**
     * get column heading
     *
     * @throws Exception
     */
    public function headings(): array
    {
        $this->loadColumns();

        $headings = [];

        if (is_array($this->columns)) {
            $this->loadHeadings($this->columns, $headings, $this->type);
            ksort($headings);
        }

        return array_values($headings);
    }

    /**
     * @throws Exception
     */
    public function columnFormats(): array
    {
        $this->loadColumns();

        $formats = [];
        if (is_array($this->columns)) {
            $this->loadColumnFormats($this->columns, $formats, $this->type);
            ksort($formats);
        }

        // Prepare re-keyed results
        $alphaFormats = [];

        // Get first index
        $startIndex = array_key_first($formats);

        array_walk(
            $formats,
            function ($format, $key) use ($startIndex, &$alphaFormats) {
                if ($format !== null) {
                    $alphaFormats[chr(ord('A') + $key - $startIndex)] = $format;
                }
            }
        );

        return $alphaFormats;
    }

    /**
     * @return array
     */
    public function drawings(): array
    {
        // TODO
        return [];
    }

    /**
     * @return Closure[]
     */
    public function registerEvents(): array
    {
        return [
            BeforeExport::class => function (BeforeExport $event) {
                $event->writer->getProperties()->setCreator(config('app.name'));
                $event->writer->getProperties()->setLastModifiedBy(config('app.name'));
                $options = Schema::getModelOptions($this->type);
                $event->writer->getProperties()->setTitle('Export ['.$options['title'][config('app.locale')].']');
                $event->writer->getProperties()->setCompany(config('app.name'));
            },
            AfterSheet::class   => function (AfterSheet $event) {
                $cellRange = 'A1:'.(chr(ord('A') + (count((array) $this->columns) - 1))).'1';
                $event->sheet->getDelegate()->getStyle($cellRange)->getFont()->setSize(14);
            },
        ];
    }

    public function getCsvSettings(): array
    {
        return [
            'delimiter'              => $this->config['format']['options']['delimiter'] ?? ',',
            'enclosure'              => $this->config['format']['options']['enclosure'] ?? '"',
            'line_ending'            => $this->config['format']['options']['line_ending'] ?? PHP_EOL,
            'use_bom'                => $this->config['format']['options']['use_bom'] ?? true,
            'include_separator_line' => $this->config['format']['options']['include_separator_line'] ?? false,
            'excel_compatibility'    => $this->config['format']['options']['excel_compatibility'] ?? false,
        ];
    }
}
