<?php

namespace Inside\Content\Services;

use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;
use Inside\Content\Contracts\ContentBlueprints as ContentBlueprintsContract;
use Inside\Content\Facades\Schema;
use Inside\Content\Services\Traits\HasConsole;
use Inside\Host\Bridge\BridgeContentType;

class ContentBlueprints implements ContentBlueprintsContract
{
    use HasConsole;

    private ?BridgeContentType $bridge = null;

    private string $configurationPath;

    public function __construct()
    {
        $this->configurationPath = config('blueprints.path');
    }

    private function getBridge(): BridgeContentType
    {
        return $this->bridge ??= new BridgeContentType($this->getConsole());
    }

    public function isBundleInstalled(string $bundle): bool
    {
        return collect(config("blueprints.bundles.$bundle", []))->every(fn (string $type) => Schema::hasModel($type));
    }

    public function generateBundle(string $bundle, array $overrides = [], bool $deleteBefore = false): void
    {
        $bundleTypes = config("blueprints.bundles.$bundle");

        if (empty($bundleTypes) || ! is_array($bundleTypes)) {
            throw new \Exception("Invalid configurations for bundle $bundle. Must be an array of contentTypes.");
        }

        foreach ($bundleTypes as $contentType) {
            $this->generateContent(
                contentType: $contentType,
                overrides: $overrides[$contentType] ?? [],
                deleteBefore: $deleteBefore,
                bundle: $bundle
            );
        }
    }

    public function generateContent(string $contentType, array $overrides = [], bool $deleteBefore = false, ?string $bundle = null): void
    {
        if (! $this->hasContentConfiguration($contentType, $bundle)) {
            throw new \Exception("Missing blueprint configuration for contentType $contentType");
        }

        $className = Str::studly($contentType);

        try {
            $contentDescription = include_once $this->getContentBlueprintFilePath($contentType, $bundle);
            $contentDescription = array_merge_recursive($contentDescription, $overrides);

            if ($deleteBefore && Schema::hasModel($contentType)) {
                $this->getConsole()?->writeln("<info>Suppression du type de contenu</info> <class>$className</class>");
                $this->getBridge()->contentTypeDelete($contentType);
            }

            if (Schema::hasModel($contentType)) {
                $this->getConsole()?->writeln("<info>Mis à jour du type de contenu </info> <class>$className</class>");
            } else {
                $this->getConsole()?->writeln("<info>Création du type de contenu </info> <class>$className</class>");
            }

            $this->getBridge()->contentTypeUpdateOrCreate(
                $contentType,
                $contentDescription['options'],
                $contentDescription['fields'],
                $contentDescription['groups'] ?? [],
            );

            $this->getConsole()?->writeln("<info>Type de contenu <class>$className</class> correctement crée/mis à jour</info>");
        } catch (\Exception $e) {
            $this->getConsole()?->writeln("<error>La class <class>$className</class> n'a pu être créée/mise à jour *".$e->getMessage().'*</error>');
            throw $e;
        }
    }

    public function hasContentConfiguration(string $contentType, ?string $bundle = null): bool
    {
        return File::exists($this->getContentBlueprintFilePath($contentType, $bundle));
    }

    public function getContentBlueprintFilePath(string $contentType, ?string $bundle = null): string
    {
        if (! empty($bundle)) {
            return "$this->configurationPath/bundles/$bundle/$contentType.php";
        }

        return "$this->configurationPath/contents/$contentType.php";
    }
}
