<?php

namespace Inside\Content\Services\ConfigurationGenerator\Bundle\Field\Field;

use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Inside\Content\Services\ConfigurationGenerator\Bundle\BaseBundle;

abstract class BaseField extends BaseBundle
{
    public const ContentEntityType = 'field';

    public const Bundle = 'field';

    public const Storage = '';

    protected function __construct(
        protected BaseBundle $bundle,
        protected string $id,
        protected string $fieldName,
        protected string $label,
        protected string $description,
        protected bool $required,
        protected array $settings,
        protected string|array $defaultValue,
        protected array $thirdPartySettings,
    ) {
        parent::__construct();
    }

    abstract public static function create(BaseBundle $bundle, string $id, string $fieldName, Collection $configuration): self;

    public static function getStorage(): string
    {
        return static::Storage;
    }

    protected function getName(): string
    {
        return static::dot(static::ContentEntityType, static::Bundle, $this->bundle::Type, $this->id, $this->fieldName);
    }

    public function mapping(): array
    {
        return [
            'uuid' => $this->uuid,
            'id' => static::dot($this->bundle::Type, $this->id, $this->fieldName),
            'field_name' => $this->fieldName,
            'entity_type' => $this->bundle::Type,
            'bundle' => $this->id,
            'label' => static::translate()->from($this->label),
            'field_type' => static::Storage::Type,
            'default_value' => $this->defaultValue,
            'required' => $this->required,
            'settings' => static::translate()->all($this->settings),
            'langcode' => static::translate()->getMain(),
            'description' => static::translate()->from($this->description),
            'third_party_settings' => $this->thirdPartySettings,
            'dependencies.module' => $this->getModuleDependencies(),
            'dependencies.config' => $this->getConfigDependencies(),
        ];
    }

    public function mappingTranslation(string $langcode): array
    {
        return [
            'label' => static::translate()->from($this->label, $langcode),
            'description' => static::translate()->from($this->description, $langcode),
            'settings' => static::translate()->all($this->settings, $langcode),
        ];
    }

    protected function getModuleDependencies(): array
    {
        $modules = collect([$this->bundle::ContentEntityType]);

        if (! empty($this->thirdPartySettings)) {
            $modules = collect($this->thirdPartySettings)->keys()->merge($modules);
        }

        if (static::Storage::Module !== 'core') {
            $modules->push(static::Storage::Module);
        }

        return $modules->sort()->unique()->values()->toArray();
    }

    protected function getConfigDependencies(): array
    {
        $config = collect([
            static::dot('field', 'storage', $this->bundle::Type, $this->fieldName),
            static::dot($this->bundle::ContentEntityType, $this->bundle::Bundle, $this->id),
        ]);

        $settings = collect(Arr::dot($this->settings));
        $handler = (string) str($settings->get('handler', ''))->after('default:');

        if (empty($handler) || $handler === 'user') {
            return $config->toArray();
        }

        $handler = match ($handler) {
            'paragraph' => 'paragraphs.paragraphs_type',
            default => 'node.type'
        };

        $settings
            ->filter(fn (mixed $value) => is_string($value) && ! empty($value))
            ->each(function (string $value, string $key) use ($config, $handler) {
                if (str($key)->startsWith('handler_settings.target_bundles.') && is_string($value)) {
                    $config->push(static::dot($handler, $value));
                }
            });

        return $config->sort()->unique()->values()->toArray();
    }
}
