<?php

namespace Inside\Content\Services;

use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\ValidationException;
use Inside\Content\Facades\Schema;
use Inside\Content\Jobs\DeleteContent;
use Inside\Content\Models\Content;
use Inside\Content\Models\Field;
use Inside\Content\Models\Pivot;

/**
 * Class CleanContentService
 */
class CleanContentService
{
    /**
     * Validate content
     *
     * @throws ValidationException
     */
    public function validate(Content $content): void
    {
        $options = Schema::getModelOptions(class_to_type($content));

        if (! isset($options['deletion_strategy']) || $options['deletion_strategy'] != 'validation') {
            return;
        }

        // Count children
        if (($count = $this->getChildrenCount($content)) > 0) {
            throw ValidationException::withMessages(
                [
                    'deletion' => trans_choice('validation.deletion.havechildren', $count, ['children' => $count]),
                ]
            );
        }
    }

    /**
     * Get parent reference fields for $content
     */
    public function getParentReferenceFields(Content $content): array
    {
        return Schema::getFieldListing(
            class_to_type($content),
            function ($options) {
                return isset($options['type']) && $options['type'] == 'reference'
                  && isset($options['options']['reference_is_a_parent'])
                  && $options['options']['reference_is_a_parent'];
            }
        );
    }

    /**
     * Get children reference fields for $content
     */
    public function getChildrenReferenceFields(Content $content): Collection
    {
        $fields = Field::query()
          ->whereJsonContains('options->target', $content->content_type)
          ->where(fn (Builder $query) => $query
            ->whereJsonContains('options->reference_is_a_parent', 1)
            ->orWhereJsonContains('options->reference_is_a_parent', true)
          )
          ->pluck('name');

        if (Schema::hasField(class_to_type($content), 'comments')) {
            $fields->push(class_to_type($content));
        }

        return $fields->filter()->unique();
    }

    /**
     * Get children count
     */
    public function getChildrenCount(Content $content): int
    {
        return Pivot::query()
          ->where('related_uuid', $content->uuid)
          ->where('related_type', get_class($content))
          ->where('related_langcode', $content->langcode)
          ->whereIn('related_field', $this->getChildrenReferenceFields($content))
          ->count();
    }

    /**
     * get children for $content
     */
    public function getChildren(Content $content, bool $onlyParent = false): Collection
    {
        return Pivot::query()
          ->with(['toContent', 'fromContent'])
          ->where('related_uuid', $content->uuid)
          ->where('related_type', get_class($content))
          ->where('related_langcode', $content->langcode)
          ->whereIn('related_field', $this->getChildrenReferenceFields($content))
          ->whereHasMorph('fromContent', '*')
          ->when($onlyParent, fn (Builder $query) => $query
            ->whereDoesntHaveMorph('toContent', '*', fn (Builder $query) => $query
              ->where('related_uuid', '<>', $content->uuid)
              ->where('related_type', get_class($content))
            )
          )
          ->orderBy('weight')
          ->get()
          ->pluck('parent');
    }

    /**
     * clean a content
     */
    public function clean(Content $content, bool $isChild = false): void
    {
        $options = Schema::getModelOptions(class_to_type($content));
        $deletionStrategy = $options['deletion_strategy'] ?? null;

        if (
            $this->shouldDeleteChildren($content)
            && isset($options['deletion_strategy'])
            && in_array($deletionStrategy, ['parent', 'only_parent'])
        ) {
            $this
              ->getChildren($content, $deletionStrategy == 'only_parent')
              ->each(function (Content $child) {
                  $this->forceChildrenDeletion($child, true);
                  (new DeleteContent($child->content_type, (string) $child->uuid))->handle();
              });
        } else {
            $this->cleanPivots($content);
            $this->cleanFiles($content);
            $this->cleanSections($content);
        }
    }

    /**
     * Clean pivot
     *
     * @throws Exception
     */
    protected function cleanPivots(Content $content): void
    {
        $deleted = Pivot::query()
          ->where(fn (Builder $query) => $query
            ->where('related_type', get_class($content))
            ->where('related_uuid', $content->uuid)
          )
          ->orWhere(fn (Builder $query) => $query
            ->where('parent_type', get_class($content))
            ->where('parent_uuid', $content->uuid)
          )
          ->delete();

        Log::debug('[CleanContentService::cleanPivots] ('.$deleted.')');
    }

    /**
     * clean Files for content
     */
    protected function cleanFiles(Content $content): void
    {
        $fieldNames = Schema::getFieldListing(
            class_to_type($content),
            function ($field) {
                return isset($field['type']) && in_array($field['type'], ['image', 'file']);
            }
        );
        $storage = Storage::disk('local');
        foreach ($fieldNames as $fieldName) {
            if ($content->{$fieldName} !== null && $storage->exists($content->{$fieldName})) {
                $storage->delete($content->{$fieldName});
                Log::debug('[CleanContentService::cleanFiles] delete file <'.$content->{$fieldName}.'>');
            }
        }
    }

    /**
     * Clean sections
     */
    protected function cleanSections(Content $content): void
    {
        $fieldNames = Schema::getFieldListing(
            class_to_type($content),
            function ($field) {
                return isset($field['type']) && $field['type'] == 'section';
            }
        );

        if (empty($fieldNames)) {
            return;
        }

        foreach (Schema::getSectionTypes() as $section) {
            $deleted = section_type_to_class($section)::query()
              ->where('sectionable_uuid', $content->uuid)
              ->where('sectionable_type', get_class($content))
              ->delete();

            Log::debug('[CleanContentService::cleanSections] delete section ['.$section.'] ('.$deleted.')');
        }
    }

    protected function getForceChildrenDeletionCacheKey(Content $content): string
    {
        return "force-children-deletion-$content->uuid";
    }

    public function shouldDeleteChildren(Content $content): bool
    {
        return Cache::pull($this->getForceChildrenDeletionCacheKey($content), false);
    }

    public function forceChildrenDeletion(Content $content, bool $force = false): void
    {
        Cache::put($this->getForceChildrenDeletionCacheKey($content), $force);
    }
}
