<?php

namespace Inside\Content\Models\Traits;

use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Str;
use Inside\Authentication\Models\User;
use Inside\Content\Facades\ScopeLogic;
use Inside\Content\Scopes\PublishedScope;

trait Publishable
{
    public function initializePublishable(): void
    {
        $this->dates[] = $this->getPublishedAtColumn();
        $this->casts[$this->getUnpublishedAtColumn()] = 'timestamp';
    }

    public static function bootPublishable(): void
    {
        if (App::runningInConsole()) {
            return;
        }
        $user = Auth::user();
        if (! $user instanceof User) {
            return;
        }
        $type = class_to_type(get_called_class());

        // published_at is a non sens on users, comments AND menus
        if (in_array($type, ['comments', 'users']) || Str::endsWith($type, '_menus')) {
            return;
        }

        if (ScopeLogic::publishabledScopeNeeded($user, $type)) {
            static::addGlobalScope(new PublishedScope());
        }
    }

    /**
     * @return bool
     */
    public function isPublished(): bool
    {
        if (is_null($this->{$this->getPublishedAtColumn()}) && is_null($this->{$this->getUnpublishedAtColumn()})) {
            return true;
        }
        if (is_null($this->{$this->getPublishedAtColumn()})) {
            return $this->{$this->getUnpublishedAtColumn()}->lte(now());
        }
        if (is_null($this->{$this->getUnpublishedAtColumn()})) {
            return $this->{$this->getPublishedAtColumn()}->gte(now());
        }

        return $this->{$this->getUnpublishedAtColumn()}->lte(now()) &&
            $this->{$this->getPublishedAtColumn()}->gte(now());
    }

    /**
     * @return bool
     */
    public function isUnpublished(): bool
    {
        return ! $this->isPublished();
    }

    /**
     * @return string
     */
    public function getPublishedAtColumn(): string
    {
        return defined('static::PUBLISHED_AT') ? static::PUBLISHED_AT : 'published_at';
    }

    /**
     * @return string
     */
    public function getQualifiedPublishedAtColumn(): string
    {
        return $this->qualifyColumn($this->getPublishedAtColumn());
    }

    /**
     * @return string
     */
    public function getUnpublishedAtColumn(): string
    {
        return defined('static::UNPUBLISHED_AT') ? static::UNPUBLISHED_AT : 'unpublished_at';
    }

    /**
     * @return string
     */
    public function getQualifiedUnpublishedAtColumn(): string
    {
        return $this->qualifyColumn($this->getUnpublishedAtColumn());
    }

    /**
     * @return bool
     */
    public function hasUnpublishedAtColumn(): bool
    {
        return Schema::hasColumn($this->getTable(), $this->getUnpublishedAtColumn());
    }
}
