<?php

namespace Inside\Content\Listeners;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Str;
use Inside\Content\Events\ContentFullyInsertedEvent;
use Inside\Content\Models\Contents\News;
use Inside\Notify\Listeners\BaseNotificationListener;
use Inside\Notify\Models\NotificationSubscriber;

class WarnNewsFollowersListener extends BaseNotificationListener
{
    private function notifyUsers(string $via, array $conditions, News $model, array $userUuidsToExclude = []): array
    {
        $subscribers = NotificationSubscriber::query()
          ->with(['user', 'notificationType'])
          ->whereHas(
              'notificationType',
              fn (Builder $notificationType) => $notificationType
                ->where('action', 'follow')
                ->where('via', $via)
                ->whereIn('condition', $conditions)
          )
          ->whereNotIn('user_uuid', $userUuidsToExclude)
          ->get()
          ->unique('user_uuid');

        $subscribers->each(function (NotificationSubscriber $subscriber) use ($model) {
            if ($this->checkConditions($subscriber->notificationType, $model, $subscriber->user) === false) {
                return;
            }

            $this->route = $subscriber->user;
            $this->notify($subscriber->notificationType, $subscriber->user, $model);
        });

        return $subscribers->pluck('user.uuid')->toArray();
    }

    public function handle(ContentFullyInsertedEvent $event): void
    {
        $model = $event->model;

        if (! $model instanceof News) {
            return;
        }

        $fields = config('news_followers')['fields'];
        $userUuidsToExcludeWeb = $userUuidsToExcludeEmail = [];
        foreach ($fields as $field) {
            $references = $model->{Str::camel($field)};
            $conditions = [];
            foreach ($references as $reference) {
                $conditions[] = $field.':'.$reference->uuid.'|status:1';
            }
            $userUuidsToExcludeWeb = $this->notifyUsers('web', $conditions, $model, $userUuidsToExcludeWeb);
            $userUuidsToExcludeEmail = $this->notifyUsers('email', $conditions, $model, $userUuidsToExcludeEmail);
        }
    }
}
