<?php

declare(strict_types=1);

namespace Inside\Content\Listeners;

use Illuminate\Database\Query\Builder;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Inside\Content\Contracts\SchemaService;
use Inside\Content\Events\ProcessingUpdate;
use Inside\Content\Facades\Schema;

final class CleanupOldFiles extends BaseContentListener
{
    public function handle(ProcessingUpdate $event): void
    {
        /** @var SchemaService $service */
        $service = App::make(SchemaService::class);
        $content = $event->content;
        $dirtyFileFieldNames = array_intersect(
            array_keys($event->getDirty()),
            $service->getFieldListingOfType(class_to_type($content), ['image', 'file'])
        );

        $storage = Storage::disk('local');
        foreach ($dirtyFileFieldNames as $dirtyFileFieldName) {
            $filePath = $content->getOriginal($dirtyFileFieldName);

            if (is_string($filePath) && $storage->exists($filePath)) {
                $isUsedByAnother = collect(Schema::getModelsWithField($dirtyFileFieldName))
                    ->map(fn ($type) => Schema::isSectionType($type) ? section_type_to_table($type) : type_to_table($type))
                    ->map(fn ($table) => DB::table($table)->select('uuid', $dirtyFileFieldName)->where($dirtyFileFieldName, $filePath))
                    ->reduce(fn (?Builder $query, Builder $next) => is_null($query) ? $next : $query->union($next))
                    ->exists();

                if ($isUsedByAnother) {
                    Log::debug(__(
                        '[CleanupOldFiles] can\'t removing old file :path for fieldName :field on :type <:uuid> because it is used by another content/section',
                        [
                            'path' => $filePath,
                            'field' => $dirtyFileFieldName,
                            'type' => class_to_type($content),
                            'uuid' => $content->uuid ?? '<null>',
                        ]
                    ));

                    return;
                }

                Log::debug(__(
                    '[CleanupOldFiles] removing old file :path for fieldName :field on :type <:uuid>',
                    [
                        'path' => $filePath,
                        'field' => $dirtyFileFieldName,
                        'type' => class_to_type($content),
                        'uuid' => $content->uuid ?? '<null>',
                    ]
                ));

                $storage->delete($filePath);
            }
        }
    }
}
