<?php

namespace Inside\Content\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Support\Facades\Log;
use Inside\Content\Exceptions\ContentDeleteFailedException;
use Inside\Host\Bridge\BridgeContent;
use Inside\Jobs\Bus\Dispatchable;
use Inside\Jobs\Job;
use Inside\Jobs\Queue\Middleware\WithoutOverlapping;

/**
 * Class DeleteContent
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class DeleteContent extends Job
{
    /**
     * We really want to delete this content, let's give some tries !
     */
    public int $tries = 10;

    /**
     * Let's give it a break
     */
    public int $retryAfter = 5;

    public function __construct(
        protected string $type,
        protected string $contentUuid
    ) {
    }

    /**
     * handle delete content job
     */
    public function handle(): void
    {
        Log::debug('DeleteContent ['.$this->type.'] <'.$this->contentUuid.'>');
        $bridge = new BridgeContent();
        if (! $bridge->contentDelete($this->type, $this->contentUuid)) {
            if ($this->attempts() > $this->tries) {
                // Too many attempts, let's fail !
                $this->fail(
                    new ContentDeleteFailedException(
                        'DeleteContent ['.$this->type.'] <'.$this->contentUuid.'> [failed]'
                    )
                );
            } else {
                Log::debug(
                    'DeleteContent ['.$this->type.'] <'.$this->contentUuid.'> [attempt ('.$this->attempts()
                    .') of ('.$this->tries.') retry in ('.$this->retryAfter.')]'
                );
                $this->release($this->retryAfter);
            }
        }
    }

    /**
     * @return WithoutOverlapping[]
     */
    public function middleware(): array
    {
        return [new WithoutOverlapping($this->type.'-'.$this->contentUuid)];
    }
}
