<?php

declare(strict_types=1);

namespace Inside\Content\Http\Controllers\Resource;

use Exception;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Encryption\Encrypter;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Inside\Authentication\Models\User;
use Inside\Content\Contracts\QueryHelper;
use Inside\Content\Facades\Schema as InsideSchema;
use Inside\Permission\Exceptions\AuthorizationException;
use Symfony\Component\HttpFoundation\BinaryFileResponse;

final class ServeContent extends Controller
{
    /**
     * @throws Exception
     */
    public function __invoke(QueryHelper $queryService, string $token, string $params): BinaryFileResponse
    {
        /** @var ?User $user */
        $user = Auth::user();

        $encrypted = $params;
        $key = config('app.key');

        if (Str::startsWith($key, 'base64:')) {
            $key = base64_decode(substr($key, 7));
        }

        $encrypter = new Encrypter($key, 'AES-256-CBC');
        $params = json_decode($encrypter->decrypt($params), true);

        if (! $params) {
            throw AuthorizationException::create('serve', 'content');
        }

        $tokenNeedToBeFreed = false;

        // If I am connected, was these resources for me ?
        if ($user && $user->uuid != $params['userUuid']) {
            Log::debug(
                __('[ServeContent] conflict connected user (:userUuid) / userUuid(:paramUuid)',
                    [
                        'userUuid' => $user->uuid,
                        'paramUuid' => $params['userUuid'],
                    ]
                ));
            $tokenNeedToBeFreed = $this->logAs($params['userUuid']);
        }

        // Rebuild signature
        $signature = hash_hmac(
            'sha256',
            $encrypted,
            config('app.key')
        );

        // Check signature match token
        if (! hash_equals($signature, $token)) {
            throw AuthorizationException::create('serve', $params['contentType'], $params['contentUuid']);
        }

        // Auto log user if not logged
        if (is_null($user)) {
            $tokenNeedToBeFreed = $this->logAs($params['userUuid']);
        }

        try {
            $model = $queryService->findOrFail($params['contentType'], $params['contentUuid']);
        } catch (ModelNotFoundException $e) {
            if ($tokenNeedToBeFreed !== false) {
                $this->cleanToken($tokenNeedToBeFreed);
            }
            throw $e;
        }

        $disk = Storage::disk('local');

        // At this point, we got our content and we are allowed to access/read the content
        // Check file really exists
        if (! $disk->exists($params['path'])) {
            if ($tokenNeedToBeFreed !== false) {
                $this->cleanToken($tokenNeedToBeFreed);
            }
            throw new ModelNotFoundException;
        }

        // Check the path is for one of our model field
        $found = false;
        $path = $params['path'];
        $matches = [];

        if (preg_match('#styles/([a-z_]+){1}/(.*)#m', $path, $matches)) {
            $path = $matches[2];
        }

        $headers = [];

        if (Str::startsWith($path, 'wysiwyg/')) {
            return new BinaryFileResponse($disk->path($params['path']), 200, $headers);
        }

        foreach (InsideSchema::getFieldListingOfType($params['contentType'], ['file', 'image']) as $field) {
            if ($model->{$field} == $path) {
                $found = true;
                break;
            }
        }
        if (! $found) {
            if ($tokenNeedToBeFreed !== false) {
                $this->cleanToken($tokenNeedToBeFreed);
            }
            throw new ModelNotFoundException;
        }
        if ($tokenNeedToBeFreed !== false) {
            $this->cleanToken($tokenNeedToBeFreed);
        }

        if ($disk->mimeType($params['path']) == 'application/pdf') {
            $headers['Accept-Ranges'] = 'none';
        }

        // Finally serve the resource
        return new BinaryFileResponse($disk->path($params['path']), 200, $headers);
    }
}
