<?php

namespace Inside\Content\Http\Controllers\Resource;

use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Inside\Authentication\Models\User;
use Inside\Content\Facades\Schema as InsideSchema;
use Inside\Content\Models\Content;
use Inside\Facades\Package;
use Inside\Permission\Facades\Permission;
use Inside\Slug\Models\Slug;
use Inside\Statistics\Events\AnalyticViewLogEvent;
use OpenApi\Annotations as OA;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Symfony\Component\HttpFoundation\Response;

final class Serve extends Controller
{
    public function __invoke(string $slug, string $path): Response
    {
        /** @var ?User $user */
        $user = Auth::user();
        $unprotected = (config('app.disable_protected_files', false) !== false);
        $locale = null;

        // Private resource means at least be connected
        if (! $unprotected && is_null($user)) {
            return redirect('/?redirect=files/'.$slug.'/'.$path);
        }

        if (! $unprotected) {
            $locale = $user->langcode;
        }

        // Url decode path
        $path = urldecode($path); // TODO check that we can trust request path

        // To check permission, asked resource MUST be attached to a model and that model must be slugable
        // and it must have a slug in the corresponding user language
        // Or a section by its uuid

        if ($unprotected) {
            // Disable permission scope
            Permission::disableAllowedScope();
        }

        // Detect section or content
        if (preg_match('#^([a-fA-F0-9]{8}-[a-fA-F0-9]{4}-[a-fA-F0-9]{4}-[a-fA-F0-9]{4}-[a-fA-F0-9]{12}){1}#', $slug)) {
            // This is a file/image attach to a section or a non slugable content

            // Get section type
            [$type, $path] = explode('/', $path, 2);
            $sectionTypes = InsideSchema::getSectionTypes();
            $contentTypes = InsideSchema::getContentTypes();

            if (in_array($type, $sectionTypes)) {
                // Section
                $model = call_user_func(section_type_to_class($type).'::findOrFail', $slug);
                if (! $model->sectionable) {
                    // Section is orphan or its sectionable is not allowed
                    throw new ModelNotFoundException();
                }
            } else {
                if (in_array($type, $contentTypes)) {
                    // Content non slugable
                    $model = call_user_func(type_to_class($type).'::findOrFail', $slug);
                } else {
                    throw new ModelNotFoundException();
                }
            }
        } else {
            // Get slug or fail
            $slug = Slug::where('slug', $slug)->firstOrFail();

            // Get corresponding model
            $model = call_user_func(table_to_class($slug->type).'::findOrFail', $slug->uuid);
        }

        $disk = Storage::disk('local');

        // At this point, we got our content and we are allowed to access/read the content
        // Check file really exists

        if (! $disk->exists($path)) {
            throw new ModelNotFoundException();
        }
        $type = class_to_type($model);

        // Check the path is for one of our model field
        $fields = InsideSchema::getFieldListingOfType($type, ['file', 'image']);
        $fieldFound = false;
        $toSearch = $path;

        if (preg_match('#styles/([a-z0-9_]+){1}/(.*)#m', $toSearch, $matches)) {
            $toSearch = $matches[2];
        }

        foreach ($fields as $fieldName) {
            if ($model->{$fieldName} == $toSearch) {
                $fieldFound = $fieldName;
                break;
            }
            if ($model->{$fieldName} && Str::contains($toSearch, '.webp')) {
                /** @var array $info */
                $info = pathinfo($model->{$fieldName});
                $contentFilename = $info['dirname'].'/'.$info['filename'].'-'.($info['extension'] ?? '');
                if ($contentFilename.'.webp' === $toSearch) {
                    $fieldFound = $fieldName;
                    break;
                }
            }
        }
        if ($fieldFound === false) {
            foreach (config('resource.accredit', []) as $name => $callback) {
                if (is_callable($callback) && $callback($model, $user, $path)) {
                    Log::debug(
                        '[ResourceController::serve] callback ['.$name.'] accreditate model ['.$type
                        .'] for user ['.($user->email ?? 'no-email').']'
                    );
                    $fieldFound = true;
                }
            }
            if ($fieldFound === false) {
                throw new ModelNotFoundException();
            }
        }

        $headers = [];
        if ($disk->mimeType($path) == 'application/pdf') {
            $headers['Accept-Ranges'] = 'none';
        }

        if ($model instanceof Content && is_string($fieldFound)) {
            $this->logRessourceAccess($model, $fieldFound);
        }

        $statisticsConfig = config('statistics.types', []);
        if (collect(['documents', 'file', 'image'])->contains($model->content_type) && isset($statisticsConfig[$model->content_type])) {
            event(new AnalyticViewLogEvent($model->content_type, $model, $user, now()));
        }

        // Finally serve the resource
        return new BinaryFileResponse($disk->path($path), 200, $headers);
    }

    /**
     * Get a resource ( image, video, etc.. ) from protected storage and serve it to client
     * if the client is allowed to get it
     * files/{slug}/{path:.*}
     * @OA\Get(
     *      path="/files",
     *      operationId="fileResource",
     *      tags={"Contenu"},
     *      summary="Permet de récupérer un fichier resource ( image, vidéo, etc... ).",
     *     @OA\Parameter(
     *         name="slug",
     *         in="path",
     *         description="Le slug du contenu qui gère la resource ( la resource étant stocké dans un champ image,
     * fichier dans un catalogue ou un contenu )",
     *         required=true,
     *         @OA\Schema(
     *             type="string"
     *         ),
     *         example="mon-actualite"
     *     ),
     * @OA\Parameter(
     *         name="path",
     *         in="path",
     *         description="Le chemin interne du fichier resource. C'est un chemin qui arbitrairement dans inside et
    pour respecter la nomanclature mise en place par drupal sera du genre YYYY-mm/unique-name.ext.",
     *         required=true,
     *         @OA\Schema(
     *             type="string"
     *         ),
     *         example="2019-07/ytwwTw7fmVn22oCh.jpg"
     *     ),
     *
     * @OA\Response(
     *          response=200,
     *          description="La resource demandée",
     *         @OA\MediaType(
     *              mediaType="image/*",
     *              @OA\Schema(type="string",format="binary")
     *         )
     *      ),
     * @OA\Response(response=400, description="Bad request"),
     * @OA\Response(response=401, description="Not allowed")
     *  )
     */
}
