<?php

namespace Inside\Content\Http\Controllers\Resource;

use Exception;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Inside\Authentication\Facades\Authentication;
use Inside\Authentication\Models\Token;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Content;
use Inside\Permission\Facades\Permission;
use Inside\Statistics\Events\AnalyticLogEvent;
use Laravel\Lumen\Routing\Controller as BaseController;

abstract class Controller extends BaseController
{
    protected function logRessourceAccess(Content $model, string $fieldName): void
    {
        $type = class_to_type($model);

        if (! array_key_exists($type, config('statistics.types', []))) {
            return; // this is not a type that is logged for statstics
        }

        $user = Auth::user();

        // For the moment, only log for ged document access ( when no modal is available )
        if (! str_contains($type, 'documents')) {
            return;
        }

        if ($model->section_content_count > 0) {
            return; // This document has a modal so view is already sent
        }
        event(new AnalyticLogEvent('view', $model, $user, now()));
    }

    /**
     * Clean temp token
     * @throws Exception
     */
    protected function cleanToken(Token $token): void
    {
        $token->delete();
        Log::debug('[ResourceController::cleanToken] token ['.$token->id.'] is deleted');
    }

    protected function logAs(string $uuid): Token
    {
        $tokenNeedToBeFreed = false;
        $authUser = User::findOrFail($uuid);
        Log::debug('[ResourceController::logAs] User ['.$authUser->uuid.'] is temporary logged in');

        $insideToken = Authentication::logAs($authUser, 'newsletter_resources');

        // Set api token so guard may identify user as a logged user
        request()->headers->set('api-token', $insideToken->plainTextToken);

        // Don't log to drupal !
        request()->headers->set('pure-api', 'true');

        // Reset permission User
        Permission::reset($authUser);

        return $insideToken->accessToken;
    }
}
