<?php

namespace Inside\Content\Http\Controllers;

use Illuminate\Auth\AuthenticationException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inside\Authentication\Models\User;
use Inside\Content\Exceptions\ModelSchemaNotFoundException;
use Inside\Content\Facades\Schema;
use Inside\Content\Models\Contents\News;
use Inside\Content\Transformers\ContentTransformer;
use Inside\Database\Eloquent\Builder;
use Inside\Notify\Models\NotificationSubscriber;
use Inside\Notify\Models\NotificationType;
use Inside\Support\Str;
use Laravel\Lumen\Routing\Controller;

class NewsCategoriesController extends Controller
{
    private function subscribeUserToNotificationType(string $category, string $user, string $fieldName): void
    {
        $vias = config('notify.news_categories_notifications', ['web', 'email']);
        $data = [
            'web' => [
                'title' => 'notifications.follow.news.title',
                'description' => 'notifications.follow.news.description',
                'icon' => 'news',
                'text' => 'notifications.follow.news.text',
                'fields' => [
                    ['authors' => ['firstname', 'lastname']],
                    'title',
                ],
            ],
            'email' => [
                'title' => 'notifications.follow.news.title',
                'description' => 'notifications.follow.news.description',
                'mail' => [
                    'subject' => 'notifications.follow.news.subject',
                    'text' => 'notifications.follow.news.content',
                    'buttonText' => 'notifications.follow.news.buttonText',
                ],
                'view' => 'enhancedEmail',
                'fields' => [
                    ['authors' => ['firstname', 'lastname']],
                    'title',
                ],
            ],
        ];
        foreach ($vias as $via) {
            $type = NotificationType::where('via', $via)
              ->where('action', 'follow')
              ->where('event', 'Inside\Notify\Events\CustomNotificationEvent')
              ->where('condition', "$fieldName:$category|status:1")
              ->first();

            if (empty($type)) {
                $type = NotificationType::firstOrCreate(
                    [
                        'via' => $via,
                        'default' => false,
                        'event' => 'Inside\Notify\Events\CustomNotificationEvent',
                        'model' => 'Inside\Content\Models\Contents\News',
                        'action' => 'follow',
                        'type' => 'system',
                        'condition' => "$fieldName:$category|status:1",
                        'multiple' => false,
                        'language' => true,
                        'profile' => false,
                        'data' => $data[$via],
                    ]
                );
            }
            NotificationSubscriber::firstOrCreate([
                'user_uuid' => $user,
                'notification_type_id' => $type->id,
            ]);
        }
    }

    /**
     * @param $me
     * @param Request $request
     */
    private function checkErrors(User $me, Request $request): void
    {
        if (! $request->has('field') || ! $request->has('uuid')) {
            throw new \InvalidArgumentException('Missing field argument');
        }

        $fieldName = $request->get('field');
        if (! Schema::hasModel($fieldName)) {
            throw ModelSchemaNotFoundException::named($fieldName);
        }
    }

    public function follow(Request $request): JsonResponse
    {
        /** @var User $me */
        $me = Auth::user();
        $fieldName = $request->get('field');
        $this->checkErrors($me, $request);
        $uuid = $request->get('uuid');

        $class = type_to_class($fieldName);
        $category = call_user_func($class.'::findOrFail', $uuid);
        $class::where('uuid_host', $category->uuid_host)->pluck('uuid')->each(function ($uuid) use ($me, $fieldName) {
            $this->subscribeUserToNotificationType($uuid, $me->uuid, $fieldName);
        });

        return response()->json([
            'success' => true,
        ]);
    }

    /**
     * @throws AuthenticationException
     */
    public function unfollow(Request $request): JsonResponse
    {
        /** @var User $me */
        $me = Auth::user();
        $this->checkErrors($me, $request);
        $fieldName = $request->get('field');
        $uuid = $request->get('uuid');

        $class = type_to_class($fieldName);
        $category = call_user_func($class.'::findOrFail', $uuid);
        $class::where('uuid_host', $category->uuid_host)->pluck('uuid')->each(function ($uuid) use ($me, $fieldName) {
            $notificationTypes = NotificationType::whereIn('via', ['web', 'email'])
              ->where('action', 'follow')
              ->where('event', 'Inside\Notify\Events\CustomNotificationEvent')
              ->where('condition', "$fieldName:$uuid|status:1")->get();

            foreach ($notificationTypes as $notificationType) {
                NotificationSubscriber::where(
                    [
                        'user_uuid' => $me->uuid,
                        'notification_type_id' => $notificationType->id,
                    ]
                )->delete();
            }
        });

        return response()->json([
            'success' => true,
        ]);
    }

    /**
     * @throws AuthenticationException
     */
    public function isFollowing(Request $request): JsonResponse
    {
        /** @var User $me */
        $me = Auth::user();
        $this->checkErrors($me, $request);
        $fieldName = $request->get('field');
        $conditionUuid = $request->get('uuid');

        $isSubscribed = NotificationType::query()
          ->where('via', 'web')
          ->where('action', 'follow')
          ->where('event', 'Inside\Notify\Events\CustomNotificationEvent')
          ->where('model', 'Inside\Content\Models\Contents\News')
          ->where('condition', "$fieldName:$conditionUuid|status:1")
          ->whereHas('subscribers', fn (Builder $query) => $query->where('user_uuid', $me->uuid))
          ->count();

        return response()->json([
            'data' => (bool) $isSubscribed,
        ]);
    }

    public function following(Request $request): JsonResponse
    {
        /** @var User $me */
        $me = Auth::user();

        $categoriesUuids = NotificationType::query()
          ->where('via', 'web')
          ->where('action', 'follow')
          ->where('event', 'Inside\Notify\Events\CustomNotificationEvent')
          ->where('model', 'Inside\Content\Models\Contents\News')
          ->whereHas('subscribers', fn (Builder $query) => $query->where('user_uuid', $me->uuid))
          ->pluck('condition')
          ->unique()
          ->map(function ($condition) {
              preg_match('/([^:]+):([^|]+)/', $condition, $matches);

              return [
                  'field_name' => $matches[1],
                  'uuid' => $matches[2],
              ];
          })
          ->groupBy('field_name')
          ->map(function ($group) {
              return $group->pluck('uuid');
          })
          ->filter();

        $fields = config('news_followers')['fields'];
        collect($fields)->diff($categoriesUuids->keys())->each(function ($item) use ($categoriesUuids) {
            $categoriesUuids->put($item, collect([]));
        });
        $categoriesWithFollowStatus = $categoriesUuids->map(function ($uuids, $fieldName) use ($me) {
            return type_to_class($fieldName)::query()
              ->where('langcode', $me->langcode)
              ->get()
              ->map(function ($category) use ($uuids) {
                  $category->followed = in_array($category->uuid, $uuids->toArray());

                  return $category;
              });
        });

        return response()->json([
            'data' => $categoriesWithFollowStatus->all(),
        ]);
    }

    public function followingNews(Request $request): JsonResponse
    {
        /** @var User $me */
        $me = Auth::user();

        $categoriesUuids = NotificationType::query()
          ->where('via', 'web')
          ->where('action', 'follow')
          ->where('event', 'Inside\Notify\Events\CustomNotificationEvent')
          ->where('model', 'Inside\Content\Models\Contents\News')
          ->whereHas('subscribers', fn (Builder $query) => $query->where('user_uuid', $me->uuid))
          ->pluck('condition')
          ->unique()
          ->map(function ($condition) {
              preg_match('/([^:]+):([^|]+)/', $condition, $matches);

              return [
                  'field_name' => $matches[1],
                  'uuid' => $matches[2],
              ];
          })
          ->groupBy('field_name')
          ->map(function ($group) {
              return $group->pluck('uuid');
          })
          ->filter();

        if ($categoriesUuids->isEmpty()) {
            return response()->json([
                'data' => [],
            ]);
        }

        $query = News::query()->where('status', 1)->whereDate('published_at', '<=', now()->toDateString())->where('langcode', $me->langcode);
        $query->where(function ($query) use ($categoriesUuids) {
            $categoriesUuids->keys()->each(function ($key) use ($query, $categoriesUuids) {
                $query->orWhereHas(Str::camel($key), function ($query) use ($key, $categoriesUuids) {
                    $query->whereIn('uuid', $categoriesUuids->get($key));
                });
            });
        });

        /** @var string $filters */
        $filters = $request->query('filters') ?? '';
        $filters = json_decode($filters, true);
        $limit = $filters['limit'] ?? null;
        if ($limit) {
            $query->limit($limit);
        }

        $news = $query->orderBy('published_at', 'desc')->get();
        $transformer = new ContentTransformer();
        $transformedNews = $news->map(function ($model) use ($transformer) {
            return $transformer->transform($model);
        });

        return response()->json([
            'data' => $transformedNews,
        ]);
    }
}
