<?php

namespace Inside\Content\Http\Controllers;

use Exception;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Lang;
use Inside\Authentication\Models\User;
use Inside\Content\Contracts\DrupalSchema as DrupalSchemaContract;
use Inside\Content\Contracts\ModelForm as ModelFormContract;
use Inside\Content\Exceptions\ModelSchemaNotFoundException;
use Inside\Content\Facades\Schema;
use Inside\Content\Facades\Schema as InsideSchema;
use Inside\Content\Models\Model;
use Inside\Permission\Exceptions\AuthorizationException;
use Inside\Permission\Facades\Permission;
use Laravel\Lumen\Routing\Controller;

/**
 * Inside content controller.
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class FormController extends Controller
{
    /**
     * Get content schema for query builder
     */
    public function schema(DrupalSchemaContract $schema): JsonResponse
    {
        return response()->json(['data' => $schema()]);
    }

    /**
     * List available add form
     */
    public function listForm(): JsonResponse
    {
        $models = Model::all();
        $user = Auth::user();
        $forms = [];

        try {
            $user = User::findOrFail($user->uuid);
        } catch (ModelNotFoundException $e) {
            return response()->json([]);
        }

        foreach ($models as $model) {
            if (! str_contains($model->class, 'Inside\Content\Models\Sections\\')
                && isset($model->options['name'])
                && isset($model->options['title'])
            ) {
                if ($user->isSuperAdmin()) {
                    $forms[] = $model->toArray();
                } else {
                    // Only get minimum information for other users
                    $forms[] = [
                        'options' => [
                            'title' => $model->options['title'],
                            'name' => $model->options['name'],
                        ],
                    ];
                }
            }
        }

        return response()->json($forms);
    }

    /**
     * Get content form
     * @throws Exception|AuthorizationException
     */
    public function form(Request $request, ModelFormContract $form, string $type, ?string $id = null): JsonResponse
    {
        if (is_null($id) && ! Permission::allowed('create', $type)) {
            throw AuthorizationException::create('create', $type);
        } elseif (! is_null($id) && ! Permission::allowed('update', $type, $id)) {
            throw AuthorizationException::create('update', $type, $id);
        }

        $isDuplicate = $request->get('is_duplicate', false);
        /** @var ?array $data */
        $data = $form($type, $id, $isDuplicate);

        if (is_null($data)) {
            throw new Exception(__('Can not get form :type (:id)', [
                'type' => $type,
                'id' => $id ?? '',
            ]));
        }

        $response = [
            'data' => $data['form'],
            'langcode' => $data['langcode'],
            'metadata' => $data['metadata'],
        ];

        if ($type == 'users' && ! is_null($id)) {
            $user = User::findOrFail($id);
            $response['provider'] = ! $user->provider_type
                ? __('users.form.provider.manual', locale: $data['langcode'])
                : __('users.form.provider.auto', [
                    'provider_type' => $user->provider_type,
                    'provider_name' => $user->provider_name,
                ], $data['langcode']);
        }

        return response()->json($response);
    }

    /**
     * Get form field option field
     */
    public function field(string $type, string $fieldName): JsonResponse
    {
        if (! Schema::hasField($type, $fieldName)) {
            return response()->json(['data' => null]);
        }

        return response()->json(['data' => Schema::getFieldOptions($type, $fieldName)]);
    }

    public function typeFields(Request $request, string $type): JsonResponse
    {
        if (! InsideSchema::hasModel($type)) {
            throw ModelSchemaNotFoundException::named($type);
        }

        $fields = array_map(
            function ($fieldName) use ($type, $request) {
                $options = InsideSchema::getFieldOptions($type, $fieldName);

                if ($fieldName == 'author') {
                    // author field is bugged  so fix there
                    $fieldName = 'authors'; // Set to reference name instead of column name ( that should be called author_uuid )
                }

                $information = [
                    'name' => $fieldName,
                    'type' => $options['type'],
                    'title' => $options['title'][$request->user()->langcode] ?? $fieldName,
                ];

                if ($options['type'] == 'reference') {
                    $information['target'] = Arr::first($options['target']);
                }

                return $information;
            },
            InsideSchema::getFieldListing(
                $type,
                function ($field, $fieldName) use ($request) {
                    // with_hidden option
                    if ($request->get('with_hidden', true)
                        && ! $field['displayed']
                        && in_array($fieldName, ['created_at'])) {
                        return false;
                    }
                    // exportable ?
                    if ($request->get('only_exportable', false)
                        && isset($field['options']['non_exportable'])
                        && $field['options']['non_exportable']) {
                        return false;
                    }

                    if ($request->get('only_exportable', false) && in_array($fieldName, ['password'])) {
                        return false;
                    }

                    return true;
                }
            )
        );

        if ($type == 'users') {
            $fields[] = [
                'name'      =>  'roles',
                'type'      =>  'text',
                'title'     =>  Lang::get('export.titles.roles'),
            ];
        }
        // export created_at column value
        $fields[] = [
            'name'      =>  'created_at',
            'type'      =>  'timestamp',
            'title'     =>  Lang::get('export.titles.created_at'),
        ];

        return response()->json($fields);
    }
}
