<?php

namespace Inside\Content\Http\Controllers;

use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Inside\Content\Contracts\ContentTree;
use Inside\Content\Exceptions\ModelSchemaNotFoundException;
use Inside\Content\Facades\Schema;
use Inside\Content\Models\Content;
use Laravel\Lumen\Routing\Controller;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

class ContentTreeController extends Controller
{
    public function tree(ContentTree $contentTree, string $type, string $id, Request $request): JsonResponse
    {
        if (! Schema::hasModel($type)) {
            throw ModelSchemaNotFoundException::named($type);
        }

        if (! $contentTree->hasValidTreeConfiguration($type)) {
            abort(400, "Type '$type' not configured");
        }

        $content = call_user_func(type_to_class($type).'::find', $id);

        if (! $content instanceof Content) {
            throw new NotFoundHttpException('Content not found');
        }

        $depth = $request->query->getInt('depth', 1);

        if ($depth < 1) {
            abort(400, 'Invalid parameter: depth must be a positive int');
        }

        $root = $contentTree->getContentRoot($content);
        $parent = $contentTree->getContentParent($content, $depth);

        $parent = $root?->uuid === $parent?->uuid ? null : $parent;

        return response()->json([
            'content' => $contentTree->formatContent($content),
            'parent' => $contentTree->formatContent($parent),
            'root' => $contentTree->formatContent($root),
        ]);
    }
}
