<?php

declare(strict_types=1);

namespace Inside\Content\Console\Paragraphs;

use Illuminate\Support\Facades\Lang;
use Inside\Console\Command;
use Inside\Content\Facades\Schema;
use Inside\Host\Bridge\BridgeContentType;

final class FixMultilingualLabels extends Command
{
    protected $signature = 'inside:paragraph:fix:labels {--section=}?';

    protected $description = 'Add paragraphs labels translations';

    private ?BridgeContentType $bridge = null;

    public function handle(): void
    {
        $section = $this->option('section');
        $sections = $section ? [$section] : Schema::getSectionTypes();
        collect($sections)->each(fn (string $section) => $this->addSectionFieldsLabels($section));
        $this->call('models:rebuild');
    }

    private function addSectionFieldsLabels(string $section): void
    {
        $this->writeln("<fg=green>Updating section <fg=yellow> $section </>");
        $this->getBridge()->contentTypeUpdate($section, ['translatable' => true], [], [], 'paragraph');

        collect(Schema::getFieldListing($section))
            ->diff(['status', 'author'])
            ->each(function (string $field) use ($section) {
                $options = Schema::getFieldOptions($section, $field);
                $titles = $this->getTranslations("field.$field.label");

                if ($this->shouldUpdateTitles($titles, $options)) {
                    $field = $field === 'body' ? 'field_body' : $field; // Need to add field_ to body in order to bypass bridge protection for drupal core fields
                    $this->getBridge()->changeFieldTitle($section, $field, $titles, 'paragraph');
                }
            });
    }

    private function getTranslations(string $key): array
    {
        return collect(list_languages())
            ->mapWithKeys(fn (string $langcode) => [$langcode => Lang::get($key, [], $langcode)])
            ->each(function (string $translation, string $langcode) use ($key) {
                if ($key === $translation) {
                    $this->writeln("<fg=red>Missing translation <fg=yellow> $key </> for langcode <fg=yellow> $langcode </>");
                }
            })
            ->all();
    }

    private function shouldUpdateTitles(array $titles, array $fieldOptions): bool
    {
        return collect($titles)->some(fn ($title, $langcode) => $title !== ($fieldOptions['title'][$langcode] ?? null));
    }

    private function getBridge(): BridgeContentType
    {
        return $this->bridge ??= new BridgeContentType();
    }
}
