<?php

namespace Inside\Content\Console;

use Illuminate\Console\Command;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Facades\File;
use Inside\Content\Facades\Schema;
use Inside\Host\Bridge\BridgeContentType;
use Inside\Support\Str;
use ReflectionClass;

class MakeContentTypeCommand extends Command
{
    protected $name = 'inside:make:content';

    protected $signature = 'inside:make:content';

    protected $description = 'Generates content type migrations';

    public function handle(): void
    {
        $languages = list_languages();
        $date = date('Y_m_d', time()).'_000000';
        $contentTypes = [];

        do {
            $contentTypeName = $this->getUserInput('Enter the content type machine name');

            $contentTypes[$contentTypeName] = [
                'options' => [
                    'name' => $contentTypeName,
                    'translatable' => count($languages) > 1 ? ($this->getUserInput('Is your content type translatable?', ['y', 'n'], 'y') === 'y' ? true : false) : false,
                    'searchable' => $this->getUserInput('Is your content type searchable?', ['y', 'n'], 'y') === 'y' ? true : false,
                    'global_searchable' => $this->getUserInput('Is your content type global searchable?', ['y', 'n'], 'y') === 'y' ? true : false,
                    'aliasable' => $this->getUserInput('Is your content type aliasable?', ['y', 'n'], 'y') === 'y' ? true : false,
                    'permissible' => $this->getUserInput('Is your content type permissible?', ['y', 'n'], 'y') === 'y' ? true : false,
                    'categorizable' => $this->getUserInput('Is your content type categorizable?', ['y', 'n'], 'y') === 'y' ? true : false,
                    'listing_type' => $this->getUserInput('What listing type to use', ['content', 'taxo', 'hidden'], 'content'),
                    'title' => [],
                ],
                'fields' => [],
            ];

            foreach ($languages as $language) {
                $contentTypes[$contentTypeName]['options']['title'][$language] = $this->getUserInput(sprintf('Enter the %s content type title', strtoupper($language)));
            }

            $fieldWeight = 0;

            while ($this->getUserInput('Do you want to add a field?', ['y', 'n'], 'n') === 'y') {
                $field = [
                    'name' => $this->getUserInput('Enter your field machine name'),
                    'searchable' => $this->getUserInput('Is your field searchable?', ['y', 'n'], 'y') === 'y' ? true : false,
                    'translatable' => count($languages) > 1 ? ($this->getUserInput('Is your field translatable?', ['y', 'n'], 'y') === 'y' ? true : false) : false,
                    'search_result_field' => $this->getUserInput('Is your field a search result field?', ['y', 'n'], 'y') === 'y' ? true : false,
                    'searchable_filter' => $this->getUserInput('Is your field a search filter?', ['y', 'n'], 'n') === 'y' ? true : false,
                    'required' => $this->getUserInput('Is your field required?', ['y', 'n'], 'y') === 'y' ? true : false,
                    'weight' => $fieldWeight,
                    'title' => [],
                ];

                foreach ($languages as $language) {
                    $field['title'][$language] = $this->getUserInput(sprintf('Enter the %s field title', strtoupper($language)));
                }

                if ($this->getUserInput('Do you want to add a description to this field?', ['y', 'n'], 'n') === 'y') {
                    $field['description'] = [];

                    foreach ($languages as $language) {
                        $field['description'][$language] = $this->getUserInput(sprintf('Enter the %s field description', strtoupper($language)));
                    }
                }

                // ici boucle + try catch
                $fieldType = $this->getUserInput('What is your field\'s type?', ['text', 'textarea', 'integer', 'paragraphs', 'reference', 'boolean', 'image', 'file', 'timestamp', 'wysiwyg', 'link', 'color', 'icon', 'comments', 'phone']);
                $method = sprintf('get%sFieldSettings', Str::studly($fieldType));

                if (method_exists($this, $method)) {
                    $this->$method($field, $fieldWeight);
                }

                /** @var array $field */
                $field['type'] = '{{CONSTANT_'.$this->getConstantAsString($field['type']).'}}';

                if (array_key_exists('type', $field['widget'])) {
                    $field['widget']['type'] = '{{CONSTANT_'.$this->getConstantAsString($field['widget']['type']).'}}';
                }

                $contentTypes[$contentTypeName]['fields'][] = $field;

                $fieldWeight++;
            }
        } while ($this->getUserInput('Do you want to create a new content type?', ['y', 'n'], 'n') === 'y');

        foreach ($contentTypes as $name => $contentType) {
            $exportTo = cms_base_path('export/contents/'.$name);
            if (! File::exists($exportTo)) {
                File::makeDirectory($exportTo, 0755, true);
            }

            $encoder = new \Riimu\Kit\PHPEncoder\PHPEncoder();
            File::put(
                $exportTo.'/'.$date.'.php',
                '<?php
use Inside\Host\Bridge\BridgeContentType;
return '.preg_replace(
                    '#\'\{\{CONSTANT_([^\}\}]+)\}\}\'#',
                    'BridgeContentType::$1',
                    $encoder->encode(
                        $contentType,
                        [
                            'string.escape' => false,
                        ]
                    ).';'
                )
            );
        }
    }

    protected function getUserInput(string $question, array $expected = null, string $default = null): string
    {
        if (empty($expected)) {
            return $this->ask($question);
        }

        do {
            if (isset($value)) {
                $this->info(sprintf('Expected answers are [%s]', implode(', ', $expected)));
            }

            $question .= sprintf(' (%s)', implode('/', $expected));
            $value = $this->ask($question, $default);
        } while (! in_array($value, $expected));

        return $value;
    }

    /**
     * @param string $constant
     * @return int|string
     */
    protected function getConstantAsString(string $constant)
    {
        $contentTypeClass = new ReflectionClass(BridgeContentType::class);
        $constants = $contentTypeClass->getConstants();

        $constName = null;
        foreach ($constants as $name => $value) {
            if ($value == $constant) {
                return $name;
            }
        }

        throw new ModelNotFoundException();
    }

    protected function getPhoneFieldSettings(array &$field, int $fieldWeight): void
    {
        $field['widget'] = [
            'type' => BridgeContentType::INSIDE_PHONE_WIDGET,
            'weight' => $fieldWeight,
        ];
        $field['type'] = BridgeContentType::TEXT_FIELD;
        $field['settings'] = [
            'max_length' => 255,
            'is_ascii' => false,
            'case_sensitive' => false,
            'cardinality' => 1,
        ];

        if ($this->getUserInput('Do you want to add a default value to this field?', ['y', 'n'], 'n') === 'y') {
            $field['default'] = $this->getUserInput('Enter the default value');
        }
    }

    protected function getTextFieldSettings(array &$field, int $fieldWeight): void
    {
        $field['widget'] = [
            'type' => BridgeContentType::STRING_WIDGET,
            'settings' => [
                'size' => 60,
                'placeholder' => '',
            ],
            'weight' => $fieldWeight,
        ];
        $field['type'] = BridgeContentType::TEXT_FIELD;
        $field['settings'] = [
            'max_length' => 255,
            'is_ascii' => false,
            'case_sensitive' => false,
            'cardinality' => 1,
        ];

        if ($this->getUserInput('Do you want to add a default value to this field?', ['y', 'n'], 'n') === 'y') {
            $field['default'] = $this->getUserInput('Enter the default value');
        }
    }

    protected function getTextareaFieldSettings(array &$field, int $fieldWeight): void
    {
        $field['widget'] = [
            'type' => BridgeContentType::LONGTEXT_WIDGET,
            'settings' => [
                'rows' => 5,
                'placeholder' => '',
            ],
            'weight' => $fieldWeight,
        ];
        $field['type'] = BridgeContentType::TEXTAREA_FIELD;
        $field['settings'] = [
            'case_sensitive' => false,
            'cardinality' => 1,
        ];

        if ($this->getUserInput('Do you want to add a default value to this field?', ['y', 'n'], 'n') === 'y') {
            $field['default'] = $this->getUserInput('Enter the default value');
        }
    }

    protected function getIntegerFieldSettings(array &$field, int $fieldWeight): void
    {
        $field['widget'] = [
            'type' => BridgeContentType::NUMBER_WIDGET,
            'settings' => [],
            'weight' => $fieldWeight,
        ];
        $field['type'] = BridgeContentType::INTEGER_FIELD;
        $field['settings'] = [
            'cardinality' => 1,
        ];

        if ($this->getUserInput('Do you want to add a default value to this field?', ['y', 'n'], 'n') === 'y') {
            $field['default'] = (int) $this->getUserInput('Enter the default value');
        }
    }

    protected function getParagraphsFieldSettings(array &$field, int $fieldWeight): void
    {
        $field['widget'] = [
            'type' => BridgeContentType::SECTION_WIDGET,
            'settings' => [
                'title' => 'Paragraphe',
                'title_plural' => 'Paragraphes',
                'edit_mode' => 'open',
                'add_mode' => 'dropdown',
                'form_display_mode' => 'default',
                'default_paragraph_type' => '',
            ],
            'weight' => $fieldWeight,
        ];
        $field['type'] = BridgeContentType::SECTION_FIELD;
        $field['settings'] = [
            'cardinality' => -1,
        ];

        if ($this->getUserInput('Do you want to specify the paragraphs to use?', ['y', 'n'], 'n') === 'y') {
            $field['settings']['target_type'] = [];

            while ($targetType = $this->getUserInput('What paragraph type do you want to add?', array_diff(array_merge(Schema::getSectionTypes(), ['stop']), $field['settings']['target_type']))) {
                if ($targetType === 'stop') {
                    break;
                }

                $field['settings']['target_type'][] = $targetType;
            }
        } else {
            $field['settings']['target_type'] = Schema::getSectionTypes();
        }
    }

    protected function getReferenceFieldSettings(array &$field, int $fieldWeight): void
    {
        $target = $this->getUserInput('What does your field reference?');

        if ($target !== 'users') {
            $target = [$target];
        }

        $field['widget'] = [
            'settings' => [],
            'weight' => $fieldWeight,
        ];

        $widget = $this->getUserInput('What widget to use?', ['select', 'autocomplete'], 'select');

        switch ($widget) {
            case 'select':
                $field['widget']['type'] = BridgeContentType::SELECT_WIDGET;
                break;
            case 'autocomplete':
                $field['widget']['type'] = BridgeContentType::AUTOCOMPLETE_WIDGET;
                break;
        }

        $field['type'] = BridgeContentType::REFERENCE_FIELD;
        $field['settings'] = [
            'target_type' => $target,
            'cardinality' => $this->getUserInput('Is this reference an unary relation?', ['y', 'n'], 'n') === 'y' ? 1 : -1,
        ];

        if ($field['settings']['cardinality'] !== 1 && $widget === 'select') {
            $field['selectable_all'] = $this->getUserInput('Can all values be selected at once for this field?', ['y', 'n'], 'n') === 'y' ? true : false;
        }
    }

    protected function getBooleanFieldSettings(array &$field, int $fieldWeight): void
    {
        $field['widget'] = [
            'type' => BridgeContentType::BOOLEAN_CHECKBOX_WIDGET,
            'settings' => [
                'display_label' => true,
            ],
            'weight' => $fieldWeight,
        ];
        $field['type'] = BridgeContentType::BOOLEAN_FIELD;
        $field['settings'] = [
            'cardinality' => 1,
        ];

        if ($this->getUserInput('Do you want to add a default value to this field?', ['y', 'n'], 'n') === 'y') {
            $field['default'] = (bool) $this->getUserInput('Enter the default value');
        }
    }

    protected function getImageFieldSettings(array &$field, int $fieldWeight): void
    {
        $field['widget'] = [
            'type' => BridgeContentType::IMAGE_WIDGET,
            'settings' => [
                'progress_indicator' => 'throbber',
                'preview_image_style' => 'thumbnail',
            ],
            'weight' => $fieldWeight,
        ];
        $field['type'] = BridgeContentType::IMAGE_FIELD;
        $field['settings'] = [
            'uri_scheme' => 'public',
            'default_image' => [
                'uuid' => '',
                'alt' => '',
                'title' => '',
                'width' => null,
                'height' => null,
            ],
            'target_type' => 'file',
            'display_field' => false,
            'display_default' => false,
            'cardinality' => 1,
        ];
    }

    protected function getFileFieldSettings(array &$field, int $fieldWeight): void
    {
        $field['widget'] = [
            'type' => BridgeContentType::FILE_WIDGET,
            'settings' => [
                'progress_indicator' => 'throbber',
            ],
            'weight' => $fieldWeight,
        ];
        $field['type'] = BridgeContentType::FILE_FIELD;
        $field['settings'] = [
            'display_field' => false,
            'display_default' => false,
            'uri_scheme' => 'public',
            'target_type' => 'file',
            'cardinality' => 1,
        ];
    }

    protected function getTimestampFieldSettings(array &$field, int $fieldWeight): void
    {
        $field['widget'] = [
            'type' => BridgeContentType::TIMESTAMP_WIDGET,
            'settings' => [],
            'weight' => $fieldWeight,
        ];
        $field['type'] = BridgeContentType::TIMESTAMP_FIELD;
        $field['settings'] = [
            'cardinality' => 1,
        ];
    }

    protected function getWysiwygFieldSettings(array &$field, int $fieldWeight): void
    {
        $field['widget'] = [
            'type' => BridgeContentType::WYSIWYG_WIDGET,
            'settings' => [],
            'weight' => $fieldWeight,
        ];
        $field['type'] = BridgeContentType::WYSIWYG_FIELD;
        $field['settings'] = [
            'cardinality' => 1,
        ];
    }

    protected function getLinkFieldSettings(array &$field, int $fieldWeight): void
    {
        $field['widget'] = [
            'type' => BridgeContentType::INSIDE_LINK_WIDGET,
            'settings' => [],
            'weight' => $fieldWeight,
        ];
        $field['type'] = BridgeContentType::TEXT_FIELD;
        $field['settings'] = [
            'max_length' => 255,
            'is_ascii' => false,
            'case_sensitive' => false,
            'cardinality' => 1,
        ];
    }

    protected function getColorFieldSettings(array &$field, int $fieldWeight): void
    {
        $field['widget'] = [
            'type' => BridgeContentType::INSIDE_COLOR_PICKER_WIDGET,
            'settings' => [],
            'weight' => $fieldWeight,
        ];
        $field['type'] = BridgeContentType::TEXT_FIELD;
        $field['settings'] = [
            'max_length' => 255,
            'is_ascii' => false,
            'case_sensitive' => false,
            'cardinality' => 1,
        ];

        if ($this->getUserInput('Do you want to add a default value to this field?', ['y', 'n'], 'n') === 'y') {
            $field['default'] = $this->getUserInput('Enter the default value');
        }
    }

    protected function getIconFieldSettings(array &$field, int $fieldWeight): void
    {
        $field['widget'] = [
            'type' => BridgeContentType::INSIDE_ICON_PICKER_WIDGET,
            'settings' => [],
            'weight' => $fieldWeight,
        ];
        $field['type'] = BridgeContentType::TEXT_FIELD;
        $field['settings'] = [
            'max_length' => 255,
            'is_ascii' => false,
            'case_sensitive' => false,
            'cardinality' => 1,
        ];
    }

    protected function getCommentsFieldSettings(array &$field, int $fieldWeight): void
    {
        $field['widget'] = ['hidden' => true];
        $field['type'] = BridgeContentType::COMMENT_FIELD;
        $field['settings'] = [
            'comment_type' => 'comments',
            'cardinality' => 1,
        ];
    }
}
