<?php

declare(strict_types=1);

namespace Inside\Content\Console;

use Illuminate\Support\Collection;
use Inside\Console\Command;
use Inside\Content\Contracts\WelcomeToTheJungleApiClient;
use Inside\Content\Contracts\WelcomeToTheJungleContentsManager;
use Inside\Content\Models\Contents\JobOffers;
use Inside\Content\Models\Contents\WttjApplications;
use Inside\Host\Bridge\BridgeContent;

final class ImportWTTJJobOffersCommand extends Command
{
    protected $name = 'inside:import:wttj:job_offers';

    protected $signature = 'inside:import:wttj:job_offers {--lang=*}';

    private WelcomeToTheJungleApiClient $welcomeToTheJungleApiClient;

    private WelcomeToTheJungleContentsManager $welcomeToTheJungleContentsManager;

    private BridgeContent $bridge;

    public function handle(WelcomeToTheJungleApiClient $welcomeToTheJungleApiClient, WelcomeToTheJungleContentsManager $welcomeToTheJungleContentsManager): void
    {
        if (! $welcomeToTheJungleContentsManager->isWTTJEnabled()) {
            throw new \Exception('The module WTTJ is not activated.');
        }

        $this->welcomeToTheJungleApiClient = $welcomeToTheJungleApiClient;
        $this->welcomeToTheJungleContentsManager = $welcomeToTheJungleContentsManager;

        $langcodes = collect($this->option('lang'))
            ->intersect(collect(list_languages()))
            ->whenEmpty(fn (Collection $collection) => $collection->push(config('app.locale')));

        foreach ($langcodes as $langcode) {
            $applications = WttjApplications::where('langcode', $langcode)->get();

            foreach ($applications as $application) {
                $imported = $this->importJobOffers($application, $langcode);
                $this->unPublishOldJobOffers($application, $imported);
            }
        }
    }

    private function getBridge(): BridgeContent
    {
        return $this->bridge ??= new BridgeContent();
    }

    private function importJobOffers(WttjApplications $application, string $langcode): array
    {
        $this->info("Import des offres d'emploi depuis la plateforme « $application->title » (langcode: $langcode)");
        $jobOffers = $this->welcomeToTheJungleApiClient->getJobOffers($application->wttj_reference, $langcode);

        $imported = collect();
        $this->output->progressStart(count($jobOffers));
        foreach ($jobOffers as $jobOffer) {
            $imported->push($this->welcomeToTheJungleContentsManager->createOrUpdateJobOffer($jobOffer));
            $this->output->progressAdvance();
        }
        $this->output->progressFinish();

        return $imported->filter()->values()->all();
    }

    public function unPublishOldJobOffers(WttjApplications $application, array $imported): void
    {
        $application->refresh();
        $oldJobOffers = $application->reverseJobOffers->reject(fn (JobOffers $jobOffer) => in_array($jobOffer->uuid, $imported) || ! $jobOffer->status);

        if ($oldJobOffers->isEmpty()) {
            return;
        }

        $this->info("Archivage des anciennes offres d'emploi de la plateforme « $application->title »");
        $this->output->progressStart($oldJobOffers->count());
        $oldJobOffers->each(function (JobOffers $jobOffer) {
            $this->getBridge()->contentUpdate($jobOffer->content_type, [
                'uuid' => $jobOffer->uuid,
                'status' => false,
            ]);

            $this->output->progressAdvance();
        });
        $this->output->progressFinish();
    }
}
