<?php

namespace Inside\Content\Calendars;

use Illuminate\Support\Facades\Storage;
use Inside\Content\Contracts\CalendarInterface;
use Inside\Content\Jobs\DeleteFolder;
use Inside\Support\Str;

final class Ics implements CalendarInterface
{
    public function generate(string $title, ?string $location, int $startDate, ?int $endDate, ?string $description, ?string $url): string
    {
        $start = date('Ymd\THis\Z', $startDate);
        $end = $endDate ? date('Ymd\THis\Z', $endDate) : null;
        $description = $description ? str_replace(["\r\n", "\r", "\n"], '\n', $description) : '';

        $ics = implode("\n", [
            'BEGIN:VCALENDAR',
            'VERSION:2.0',
            'PRODID:-//hacksw/handcal//NONSGML v1.0//EN',
            'BEGIN:VEVENT',
            'X-WR-TIMEZONE:Europe/Paris',
            "DTSTART:$start",
            "DTEND:$end",
            'SUMMARY:'.$title,
            "LOCATION:$location",
            "DESCRIPTION:$description",
            "URL;VALUE=URI:$url",
            'END:VEVENT',
            'END:VCALENDAR',
        ]);

        return $this->getIcsFileUrl($title, $ics);
    }

    private function getIcsFileUrl(string $title, string $ics): string
    {
        $slug = Str::slug($title, '_');
        $fileName = "$slug.ics";
        $folderName = Str::random(32);
        $finalFolderPath = "downloads/$folderName";
        $finalPath = "$finalFolderPath/$fileName";

        Storage::disk('protected')->put($finalPath, $ics);

        DeleteFolder::dispatch('protected', $finalFolderPath)->delay(now()->addMinute());

        return Storage::disk('protected')->url($finalPath);
    }
}
