<?php

namespace Inside\Content\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Lang;
use Inside\Content\Facades\ContentBlueprints;
use Inside\Content\Facades\Schema;
use Inside\Content\Models\Content;
use Inside\Host\Bridge\BridgeContent;

class WelcomeToTheJungleSeeder extends Seeder
{
    private BridgeContent $bridge;

    private function getBridge(): BridgeContent
    {
        return $this->bridge ??= new BridgeContent();
    }

    private function createOrUpdateContent(string $contentType, array $data, string $identifier = 'title'): ?Content
    {
        $existing = type_to_class($contentType)::query()
            ->where('langcode', $data['langcode'])
            ->where($identifier, $data[$identifier])
            ->first();

        if ($existing instanceof Content) {
            $data['uuid'] = $existing->uuid;
        }

        $uuid = $this->getBridge()->contentInsert($contentType, $data, ! isset($data['uuid']));

        return type_to_class($contentType)::find($uuid);
    }

    private function generateJobOffersFormTemplate(): void
    {
        $data = config('wttj.form_template.data');

        if ($data['inputs'] === null) {
            $inputsPath = config('wttj.form_template.inputs_path');
            $inputs = file_get_contents($inputsPath);

            if (! $inputs) {
                throw new \Exception("Invalid configuration for wttj.form_template_inputs_path ($inputsPath)");
            }

            $data['inputs'] = json_encode(json_decode($inputs)); // in order to delete whitespaces
        }

        $locale = config('app.locale');
        $data['title'] = Lang::get('job_offers.wttj.form_template.title', [], $locale);
        $data['langcode'] = $locale;
        $original = $this->createOrUpdateContent('forms', $data, 'job_offers_form_template');

        foreach (list_languages() as $language) {
            if ($language === $locale) {
                continue;
            }

            $data['title'] = Lang::get('job_offers.wttj.form_template.title', [], $language);
            $data['uuid_host'] = $original?->uuid_host;
            $data['langcode'] = $language;
            $this->createOrUpdateContent('forms', $data, 'job_offers_form_template');
        }
    }

    private function generateWTTJApplications(): void
    {
        $applications = config('wttj.applications');
        $locale = config('app.locale');
        foreach ($applications as $application) {
            if (empty($application['reference']) || ! is_string($application['reference'])) {
                throw new \Exception('invalid configuration: application reference must be a non-empty string');
            }

            $data = [
                'wttj_reference' => $application['reference'],
                'title' => $application['title'][$locale] ?? $application['title'],
                'langcode' => $locale,
            ];

            $original = $this->createOrUpdateContent('wttj_applications', $data, 'wttj_reference');

            foreach (list_languages() as $language) {
                if ($language === $locale) {
                    continue;
                }

                $data['title'] = $application['title'][$language] ?? $application['title'];
                $data['uuid_host'] = $original?->uuid_host;
                $data['langcode'] = $language;
                $this->createOrUpdateContent('forms', $data, 'job_offers_form_template');
            }
        }
    }

    public function run(): void
    {
        if (! Schema::hasModel('forms')) {
            throw new \Exception('forms content_type has to be installed to run this seeder');
        }

        if (! Schema::hasModel('job_offers')) {
            throw new \Exception('job_offers content_type has to be installed to run this seeder');
        }

        if (empty(config('wttj.applications'))) {
            throw new \Exception('Missing configuration wttj.applications');
        }

        if (empty(config('wttj.form_template'))) {
            throw new \Exception('Missing configuration wttj.form_template');
        }

        ContentBlueprints::generateBundle('wttj');
        Artisan::call('models:rebuild');
        $this->generateJobOffersFormTemplate();
        $this->generateWTTJApplications();
    }
}
