<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Support\Facades\File;
use Inside\Content\Facades\Schema as InsideSchema;
use Inside\Host\Bridge\BridgeContent;
use Inside\Host\Bridge\BridgeContentType;
use Inside\Host\Migrations\CreateContentTypes;

class ChangeUriFieldFromTextToLongtext extends Migration
{
    use CreateContentTypes;

    public $withinTransaction = false;

    public function up(): void
    {
        $this->bootUp(__FILE__);

        $types = InsideSchema::getModelsWithField('uri');

        if (blank($types)) {
            return;
        }

        $bridge = new BridgeContentType();
        $bridgeContent = new BridgeContent();

        $options = [];
        $data = [];

        // Create a retoration point and delete every uri field
        foreach ($types as $type) {
            $data[$type] = $this->backup($type);

            $this->console->writeln("<info>[$type]</info> Retrieve current options");
            $origin = InsideSchema::getFieldOptions($type, 'uri');
            $options[$type] = $this->options($origin);

            $this->console->writeln("<info>[$type]</info> Delete old uri field");
            $bridge->contentTypeDeleteField('node', $type, 'uri');
        }

        // Now we recreate the uri field with the new options and migrate the content
        // We need to do this in two steps because we can't create a new storage field type in drupal side
        foreach ($types as $type) {
            $this->console->writeln("<info>[$type]</info> Create new uri field");
            $bridge->contentTypeCreateField($type, 'uri', $options[$type], $options[$type]['weight']);

            foreach ($data[$type] as $item) {
                $this->console->writeln("<comment>Migrate content $type - {$item['title']} - {$item['uuid']}</comment>");

                $bridgeContent->contentUpdate($type, [
                    'uuid' => $item['uuid'],
                    'title' => $item['title'],
                    'uri' => $item['uri'],
                ]);
            }
        }
    }

    private function backup(string $type): array
    {
        $data = type_to_class($type)::whereNotNull('uri')->get();
        $date = now()->format('Y-m-d_His');
        $path = storage_path("logs/{$type}_uri_{$date}.csv");

        File::put($path, $data->pluck('uri', 'uuid')->map(fn ($uri, $uuid) => "$uuid;$uri")->implode("\n"));
        $this->console->writeln("<comment>[{$type}]</comment> Backup [uuid;uri] ici: {$path}");

        return $data->toArray();
    }

    private function options(array $origin): array
    {
        return [
            'name' => 'uri',
            'type' => BridgeContentType::URI_FIELD,
            'searchable' => data_get($origin, 'searchable', false),
            'translatable' => data_get($origin, 'translatable', false),
            'search_result_field' => data_get($origin, 'search_result_field', true),
            'default' => data_get($origin, 'default', ''),
            'required' => data_get($origin, 'required', true),
            'weight' => data_get($origin, 'weight', 1),
            'title' => data_get($origin, 'title', [
                'fr' => 'Lien',
                'en' => 'Link',
                'es' => 'Enlace',
                'de' => 'Verknüpfung',
                'nl' => 'Link',
                'pt' => 'Link',
            ]),
            'description' => data_get($origin, 'description', [
                'fr' => '',
                'en' => '',
                'es' => '',
                'de' => '',
                'nl' => '',
                'pt' => '',
            ]),
            'placeholder' => data_get($origin, 'placeholder', [
                'fr' => 'https//example.com',
                'en' => 'https//example.com',
                'es' => 'https//example.com',
                'de' => 'https//example.com',
                'nl' => 'https//example.com',
                'pt' => 'https//example.com',
            ]),
            'widget' => [
                'type' => BridgeContentType::INSIDE_LINK_WIDGET,
            ],
        ];
    }
}
