<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Support\Carbon;
use Illuminate\Support\Str;
use Inside\Content\Facades\Schema;
use Inside\Database\Migrations\HasConsole;
use Inside\Host\Bridge\BridgeContent;

class MigrateInternalLinkToRelative extends Migration
{
    use HasConsole;

    public function up(): void
    {
        $this->updateAllLinkToRelativeOnlyOnInsideSide();
    }

    public function down(): void
    {
    }

    protected function updateAllLinkToRelativeOnlyOnInsideSide(): void
    {
        foreach (Schema::getContentTypes() as $type) {
            $this->writeln('Modification des champs liens du type ('.$type.')');
            $fieldNames = Schema::getFieldListing($type, function ($options) {
                return $options['type'] == 'text' && isset($options['widget'])
                    && $options['widget'] == 'inside_link_field_widget';
            });
            if (empty($fieldNames)
                && ! Schema::hasFieldOfType($type, 'section')
            ) {
                $this->writeln('<fg=green>'.str_repeat('_', 80).'</>');
                continue;
            }

            call_user_func(
                type_to_class($type).'::each',
                function (Inside\Content\Models\Content $content) use ($type, $fieldNames) {
                    if (! $content->updated_at) {
                        return;
                    }
                    $data = [
                        'updated_at' => Carbon::createFromTimestampUTC($content->updated_at)->toDateTimeString(),
                        // We don't want to update this!
                    ];
                    foreach ($fieldNames as $fieldName) {
                        if (is_null($content->{$fieldName}) || empty($content->{$fieldName})) {
                            continue;
                        }
                        $data[Schema::getOriginalFieldName($fieldName)]
                            = $this->makeRelative((string) $content->{$fieldName});
                    }
                    if (count($data) > 1) {
                        $this->write('<fg=yellow>Mise-à-jour des champs liens du contenu (<fg=magenta>'.$content->title.'</>) [<fg=cyan>'.$type.'</>]</>');
                        $this->writeResult($content->update($data));
                    }

                    // Manage section
                    $contentSectionFieldNames = Schema::getFieldListingOfType($type, 'section');
                    foreach ($contentSectionFieldNames as $contentSectionFieldName) {
                        foreach ($content->{Str::snake('section_'.$contentSectionFieldName)} as $section) {
                            $sectionType = class_to_type($section);

                            $sectionFieldNames = Schema::getFieldListing($sectionType, function ($options) {
                                return $options['type'] == 'text' && isset($options['widget']) && $options['widget'] == 'inside_link_field_widget';
                            });
                            if (empty($sectionFieldNames)) {
                                continue; // No link field
                            }

                            $dataSection = [];
                            foreach ($sectionFieldNames as $sectionFieldName) {
                                $sectionFieldOptions = Schema::getFieldOptions($sectionType, $sectionFieldName);
                                if ($sectionFieldOptions['type'] == 'text' && isset($sectionFieldOptions['widget']) && $sectionFieldOptions['widget'] == 'inside_link_field_widget') {
                                    $dataSection[Schema::getOriginalFieldName($sectionFieldName)] = $this->makeRelative((string) $section->{$sectionFieldName});
                                }
                            }
                            if (empty($dataSection)) {
                                return;
                            }
                            $this->write('<fg=yellow>Mise-à-jour des champs d\'un paragraphe [<fg=cyan>'.$sectionType.'</>]</>');
                            $this->writeResult($section->update($dataSection));
                        }
                    }
                }
            );
            $this->writeln('<fg=green>'.str_repeat('_', 80).'</>');
        }
    }

    protected function updateAllLinkToRelativeUsingBridgeContent(): void
    {
        foreach (Schema::getContentTypes() as $type) {
            $fieldNames = Schema::getFieldListing($type, function ($options) {
                return $options['type'] == 'text' && isset($options['widget'])
                    && $options['widget'] == 'inside_link_field_widget';
            });
            if (empty($fieldNames)
                && ! Schema::hasFieldOfType($type, 'section')
            ) {
                continue;
            }

            $bridge = new BridgeContent();
            call_user_func(
                type_to_class($type).'::each',
                function ($content) use ($type, $fieldNames, $bridge) {
                    $data = [
                        'uuid' => $content->uuid,
                        'type' => 'node',
                        'bundle' => $type,
                        'updated_at' => Carbon::createFromTimestampUTC($content->updated_at)
                            ->toDateTimeString(),
                        // We don't want to update this!
                    ];
                    foreach ($fieldNames as $fieldName) {
                        if (is_null($content->{$fieldName})
                            || empty($content->{$fieldName})
                        ) {
                            continue;
                        }
                        $data[Schema::getOriginalFieldName($fieldName)]
                            = $this->makeRelative((string) $content->{$fieldName});
                    }

                    // Manage section
                    $sectionFieldNames = Schema::getFieldListingOfType(
                        $type,
                        'section'
                    );
                    foreach ($sectionFieldNames as $sectionFieldName) {
                        $sections = $content->{Str::snake('section_'
                            .$sectionFieldName)};
                        $data[$sectionFieldName] = [];
                        foreach ($sections as $section) {
                            $sectionType = class_to_type($section);
                            $dataSection = [
                                'pgID' => $section->uuid,
                                'bundle' => $sectionType,
                            ];
                            foreach (
                                Schema::getFieldListing($sectionType) as
                                $sectionField
                            ) {
                                $sectionFieldOptions
                                    = Schema::getFieldOptions(
                                        $sectionType,
                                        $sectionField
                                    );
                                $value = $section->{$sectionField};
                                if ($sectionFieldOptions['type'] == 'text'
                                    && isset($sectionFieldOptions['widget'])
                                    && $sectionFieldOptions['widget']
                                    == 'inside_link_field_widget'
                                ) {
                                    $value
                                        = $this->makeRelative((string) $section->{$sectionField});
                                }
                                $dataSection[Schema::getOriginalFieldName($sectionField)]
                                    = $value;
                            }
                            $data[$sectionFieldName][] = $dataSection;
                        }
                    }

                    $bridge->contentUpdate($type, $data);
                }
            );
        }
    }

    protected function makeRelative(string $url): ?string
    {
        return preg_replace('#^'.trim(config('app.url'), '/').'#', '', $url);
    }
}
