<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Str;
use Inside\Content\Exceptions\FieldSchemaNotFoundException;
use Inside\Content\Facades\Schema as InsideSchema;
use Inside\Content\Models\Content;
use Inside\Host\Bridge\BridgeContentType;
use Inside\Host\Migrations\CreateContentTypes;
use Symfony\Component\Console\Helper\ProgressBar;

class AddPublishedAtColumn extends Migration
{
    use CreateContentTypes;

    public function up(): void
    {
        $this->bootUp(__FILE__);
        $console = $this->getConsole();

        foreach (InsideSchema::getContentTypes() as $type) {
            if (Str::endsWith($type, '_menus') || in_array($type, ['comments', 'users'])) {
                continue;
            }
            $table = type_to_table($type);
            if (Schema::hasColumn((string) $table, 'published_at')) {
                $console->writeln('<note>Type <fg=cyan>'.$type.
                    '</> already has a <fg=magenta>published_at</> field!!!</>');
                continue;
            }
            $console->writeln('<info>Adding published_at column to type <fg=cyan>'.$type.'</></info>');

            // Add column to our content
            try {
                Schema::table((string) $table, function (Blueprint $table) {
                    $table->timestamp('published_at')->after('updated_at');
                });
            } catch (Exception $exception) {
                Log::error('Failed to create published_at field on type ['.$type.']');
            }

            $console->writeln('<info>Copying <fg=cyan>created_at</> to <fg=magenta>publised_at</> internal type to timestamp</info>');
            DB::table($table)->update(['published_at' => DB::raw('created_at')]);

            $console->writeln('<info>Adding author_id column to type <fg=cyan>'.$type.'</></info>');
            // Add column
            Schema::table(
                (string) $table,
                function (Blueprint $table) {
                    $table->uuid('author_id')->nullable()->after('author');
                }
            );

            // Copy author to author_id
            DB::table($table)->update(['author_id' => DB::raw('author')]);
            Schema::table(
                (string) $table,
                function (Blueprint $table) {
                    $table->uuid('author_id')->nullable(false)->change();
                }
            );
        }
        $this->finishUp();
    }

    public function down(): void
    {
    }
}
