<?php

namespace Inside\Authentication\Services;

use Exception;
use Inside\Authentication\Exceptions\TwoFactorAuthenticationException;
use Inside\Authentication\Facades\SfrSmsSender;
use Inside\Authentication\Models\OtpToken;
use Inside\Authentication\Models\User;
use Inside\Support\Str;

class TwoFactorAuthenticationService
{
    /**
     * @param User $user
     * @return mixed
     * @throws TwoFactorAuthenticationException
     */
    public function sendOtpToken(User $user): mixed
    {
        // Check if the user has phone number
        $phoneNumberFieldName = env('2FA_PHONE_FIELD_NAME', 'mobile_phone');
        if (! $user->{$phoneNumberFieldName}) {
            throw TwoFactorAuthenticationException::NoPhoneNumberSet();
        }
        // Delete old otp tokens for this user if there is one
        OtpToken::whereUserUuid($user->uuid)->delete();

        // Create new otp token
        $newOtpToken = $this->generateNewOtp();

        // Insert otp token generated into database
        $otpToken = new OtpToken();
        $otpToken->token = (string) $newOtpToken;
        $otpToken->user_uuid = $user->uuid;
        $otpToken->save();

        // Send otp token to the user phone number
        return SfrSmsSender::sendSms(
            $user->{$phoneNumberFieldName},
            "Bonjour ! voici votre code d'authentification unique : ".$otpToken->token.' .'
        );
    }

    public function checkClientIpInGivenRanges(string $clientIp): bool
    {
        if (! env('2FA_AUTHORIZED_RANGE_IPS')) {
            return false;
        }

        $authorizedIps = explode(',', env('2FA_AUTHORIZED_RANGE_IPS'));
        $isAuthorized = false;

        foreach ($authorizedIps as $authorized) {
            $baseIp = Str::before($authorized, '/');
            if (! Str::startsWith($clientIp, $baseIp)) {
                continue;
            }

            $range = Str::after($authorized, '/');
            $min = (int) Str::before($range, '&');
            $max = (int) Str::after($range, '&');
            $check = (int) ltrim($clientIp, $baseIp.'.');

            if ($min <= $check && $check <= $max) {
                $isAuthorized = true;
                break;
            }
        }

        return $isAuthorized;
    }

    private function generateNewOtp(): string
    {
        $allowedNumbers = '0123456789';
        $otp = '';

        for ($otpLength = 1; $otpLength <= 6; $otpLength++) {
            $otp .= substr($allowedNumbers, (rand() % (strlen($allowedNumbers))), 1);
        }

        return $otp;
    }
}
