<?php

namespace Inside\Authentication\Services;

use Illuminate\Auth\Events\Lockout;
use Illuminate\Cache\RateLimiter;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Lang;
use Illuminate\Validation\ValidationException;
use Inside\Support\Str;

trait ThrottlesLogins
{
    /**
     * Ensure a login attempt is not rate limited !
     */
    protected function ensureIsNotRateLimited(): void
    {
        if (! $this->hasTooManyLoginAttempts()) {
            return;
        }

        $this->fireLockoutEvent();

        $this->sendLockoutResponse();
    }

    /**
     * check if user has too many login attempts !
     *
     * @return bool
     */
    protected function hasTooManyLoginAttempts(): bool
    {
        return $this->getRateLimiter()->tooManyAttempts(
            $this->getThrottleKey(),
            $this->getMaxAttempts()
        );
    }

    /**
     * Get an instance of ratelimiter
     *
     * @return RateLimiter
     */
    protected function getRateLimiter(): RateLimiter
    {
        return app(RateLimiter::class);
    }

    /**
     * Get throttlekey
     *
     * @return string
     */
    protected function getThrottleKey(): string
    {
        $request = $this->getRequest();

        /** @var string $email */
        $email = $request->input('email');

        $clientIp = $request->server('HTTP_X_FORWARDED_FOR') ?? $request->ip();

        return Str::lower($email).'|'.$clientIp;
    }

    protected function getRequest(): Request
    {
        return request();
    }

    /**
     * Max failed attempts !
     *
     * @return int
     */
    public function getMaxAttempts(): int
    {
        return property_exists($this, 'maxAttempts')
            ? $this->maxAttempts
            : config(
                'auth.login_throttles.max_attempts',
                5
            );
    }

    /**
     * Fire lockout event
     */
    protected function fireLockoutEvent(): void
    {
        event(new Lockout($this->getRequest()));
    }

    /**
     * Send lockout exception !
     *
     * @throws ValidationException
     */
    protected function sendLockoutResponse(): void
    {
        $seconds = $this->getRateLimiter()->availableIn(
            $this->getThrottleKey()
        );

        throw ValidationException::withMessages(
            [
                'email' => [
                    Lang::get(
                        'auth.throttle',
                        [
                            'seconds' => $seconds,
                            'minutes' => ceil($seconds / 60),
                        ]
                    ),
                ],
            ]
        )->status(Response::HTTP_TOO_MANY_REQUESTS);
    }

    /**
     * increment a failed login attempts !
     */
    protected function incrementLoginAttempts(): void
    {
        $this->getRateLimiter()->hit(
            $this->getThrottleKey(),
            $this->getDecayMinutes()
        );
    }

    /**
     * Get the number of minutes before unlocking
     *
     * @return int
     */
    public function getDecayMinutes(): int
    {
        return property_exists($this, 'decayMinutes')
            ? $this->decayMinutes
            : config(
                'auth.login_throttles.decay_minutes',
                1
            );
    }

    /**
     * Clear & reset login attempts
     */
    protected function clearLoginAttempts(): void
    {
        $this->getRateLimiter()->clear($this->getThrottleKey());
    }
}
