<?php

namespace Inside\Authentication\Services;

use Carbon\CarbonInterval;
use Exception;
use Illuminate\Contracts\Encryption\EncryptException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cookie;
use Illuminate\Support\Facades\Event;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\InteractsWithTime;
use Illuminate\Validation\ValidationException;
use Inside\Authentication\AccessToken;
use Inside\Authentication\Contracts\Authentication as AuthenticationContract;
use Inside\Authentication\Events\AuthenticationLoginEvent;
use Inside\Authentication\Events\AuthenticationLogoutEvent;
use Inside\Authentication\Events\FirstConnection;
use Inside\Authentication\Events\UserHasBeenDisconnectedEvent;
use Inside\Authentication\Exceptions\AuthenticationException;
use Inside\Authentication\Exceptions\CredentialsValidatorException;
use Inside\Authentication\Exceptions\PasswordNotValidException;
use Inside\Authentication\Exceptions\UserNotActiveException;
use Inside\Authentication\Exceptions\UserNotFoundException;
use Inside\Authentication\Facades\InsideSessionLifetime;
use Inside\Authentication\Models\Token;
use Inside\Authentication\Models\User;
use Inside\Authentication\Validators\AuthenticationValidator;
use Inside\Cookie\CookieValuePrefix;
use Inside\Host\Bridge\BridgeAuthentication;
use Inside\Kernel\Authentication\StoreIntendedPathService;
use InvalidArgumentException;
use stdClass;
use Throwable;

/**
 * Authentication service.
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class AuthenticationService implements AuthenticationContract
{
    use InteractsWithTime;
    use ThrottlesLogins;

    /**
     * Log the user in
     * @throws ValidationException
     * @throws PasswordNotValidException
     * @throws UserNotActiveException
     * @throws UserNotFoundException
     * @throws Throwable
     */
    public function login(array $credentials, bool $remember = false): AccessToken
    {
        $this->ensureIsNotRateLimited();

        try {
            $user = $this->getUserFromCredentials($credentials);
        } catch (Throwable $exception) {
            $this->incrementLoginAttempts();
            if (App::isProduction()) {
                // In production, get a more generic error message
                $this->sendFailedLoginResponse();
            }
            throw $exception;
        }
        $this->clearLoginAttempts();

        return $this->logAs($user, 'inside', false, $remember);
    }

    /**
     * Get a user with his credentials
     *
     * @throws UserNotFoundException
     * @throws PasswordNotValidException
     * @throws UserNotActiveException
     * @throws CredentialsValidatorException
     */
    protected function getUserFromCredentials(array $credentials): User
    {
        $this->validate($credentials);

        $user = null;
        if (isset($credentials['name'])) {
            $user = User::where('name', $credentials['name'])->first();
        } elseif (isset($credentials['email'])) {
            $user = User::where('email', '=', $credentials['email'])->first();
        }

        if (! $user) {
            throw new UserNotFoundException(Lang::get('auth.userDoesNotExist'));
        }

        if (! $user->status) {
            throw new UserNotActiveException(Lang::get('auth.userIsNotActive'));
        }

        if (! Hash::check($credentials['password'], $user->password)) {
            throw new PasswordNotValidException(Lang::get('auth.incorrectPassword'));
        }

        $customLoginRules = config('authentication.login_rules', []);

        foreach ($customLoginRules as $customLoginRule) {
            if ($customLoginRule && is_callable($customLoginRule)) {
                $customLoginRule($user);
            }
        }

        return $user;
    }

    /**
     * Validate user credentials format
     *
     * @throws CredentialsValidatorException
     */
    public function validate(array $credentials): void
    {
        $validator = new AuthenticationValidator();
        $validator->validate($credentials);
    }

    /**
     * @throws ValidationException|AuthenticationException
     */
    protected function sendFailedLoginResponse(): void
    {
        throw new AuthenticationException(Lang::get('auth.failed'));
    }

    /**
     * Log as $user, force to reload token if $force is true
     *
     * @throws UserNotActiveException
     */
    public function logAs(User $user, string $authenticator, bool $force = false, bool $remember = false): AccessToken
    {
        if (! $user->status) {
            throw new UserNotActiveException(Lang::get('auth.userIsNotActive'));
        }

        if ($user->currentAccessToken() === null || $force) {
            $accessToken = $user->createToken($authenticator, ['*'], $remember);
            if ($authenticator != 'inside-admin') {
                if ($user->last_login_at === null) {
                    event(new FirstConnection($user));
                }

                $user->update(['last_login_at' => now()]);
            }
        } else {
            throw new InvalidArgumentException;
        }

        // to avoid to load drupal when session is needed, we drop a cookie as well
        $this->setUserToMagicCookie(request(), $user, $accessToken->plainTextToken, Auth::check(), $remember);

        if (app()->runningInConsole()) {
            config(['auth.magic_cli_token' => $accessToken->plainTextToken]);
        }

        AuthenticationLoginEvent::dispatch($user, $accessToken->accessToken);

        Log::info('Login successfull', [
            'uuid' => $user->uuid,
            'email' => $user->email,
            'authenticator' => $authenticator,
        ]);

        return $accessToken;
    }

    /**
     * Set the usermagic cookie
     */
    public function setUserToMagicCookie(
        Request $request,
        User $user,
        string $token,
        bool $force = false,
        bool $remember = false
    ): bool {
        if (! Cookie::has('_inside_token') || $force || $this->getUserFromMagicCookie($request) === null) {
            $encrypter = app('encrypter');

            try {
                $data = [
                    'data' => $token,
                    'user_uuid' => $user->uuid,
                    'created_at' => now(),
                ];

                $data = $encrypter->encrypt($data);
            } catch (EncryptException) {
                return false;
            }
            $sessionLifeTimeInMinutes = InsideSessionLifetime::getSessionLifetime();
            if ($remember) {
                $sessionLifeTimeInMinutes = InsideSessionLifetime::getSessionLongerLifetime();
            }

            Cookie::queue('_inside_token', $data, $sessionLifeTimeInMinutes);
        }

        return true;
    }

    public function updateMagieCookieLifetime(Request $request): bool
    {
        if (! Cookie::has('_inside_token')) {
            return false;
        }
        $data = Cookie::get('_inside_token');
        $user = $this->getUserFromMagicCookie($request);
        if (! $user instanceof User
            || is_null($user->currentAccessToken())
            || ! $user->currentAccessToken()->isAboutToFade()) {
            return false;
        }

        $sessionLifeTimeInMinutes = InsideSessionLifetime::getSessionLifetime();
        if ($user->currentAccessToken()->longer_lifetime) {
            $sessionLifeTimeInMinutes = InsideSessionLifetime::getSessionLongerLifetime();
        }

        Cookie::queue('_inside_token', $data, $sessionLifeTimeInMinutes);

        return true;
    }

    /**
     * get User from request ( using cookie )
     * on a correct identification, user got a magic token, we decifer
     * that token to get current user
     */
    public function getUserFromMagicCookie(Request $request): ?User
    {
        if (! Cookie::has('_inside_token')) {
            return null; // No cookie!!
        }

        $encrypter = app('encrypter');
        $data = Cookie::get('_inside_token');
        if (! empty($data)) {
            try {
                // Disable notice error as decrypt use unserialize
                $level = error_reporting(E_ERROR);
                $data = (object) $encrypter->decrypt($data);
                error_reporting($level);
            } catch (Throwable $e) {
                $data = new stdClass();
                $data->scalar = false; // Force to parse double encrypted
            }
            if (is_object($data) && isset($data->scalar) && $data->scalar === false) {
                // Note: cookie is probably double encrypted !
                // If getUserFromMagicCookie is called from our auth guard
                // request is direct request with no middlewares and cookies are
                // not decrypted
                $data = Cookie::get('_inside_token');
                if (is_string($data)) {
                    $data = $this->doubleDecodeCookie($data);
                }
            }
            if ($data === null || ! isset($data->data)) {
                $this->invalidateUserFromMagicCookie();

                return null;
            }
            $token = Token::where('token', self::encryptToken($data->data))->first();
            if (! $token || ! $token->user || $token->user->uuid != $data->user_uuid || $token->hasExpired()) {
                $this->invalidateUserFromMagicCookie();

                return null;
            }

            return $token->user->withAccessToken(
                tap($token->forceFill(['last_used_at' => now()]))->save()
            );
        }

        return null;
    }

    /**
     * Double decode encrypted cookie data
     */
    protected function doubleDecodeCookie(mixed $data): ?stdClass
    {
        $encrypter = app('encrypter');
        try {
            $value = $encrypter->decrypt($data, false);
            $hasValidPrefix = str_starts_with($value, CookieValuePrefix::create('_inside_token', $encrypter->getKey()));
            $data = $hasValidPrefix ? CookieValuePrefix::remove($value) : $data;
            if ($data !== null && isset($data->scalar)) {
                $data = (object) $encrypter->decrypt($data->scalar);
            } else {
                $data = (object) $encrypter->decrypt($data);
            }
        } catch (Exception $e) {
            // We didn't decrypt $data!
            return null;
        }

        return $data;
    }

    /**
     * Invalidate current user magic cookie
     */
    public function invalidateUserFromMagicCookie(): void
    {
        try {
            Log::debug('[Authentication::invalidateUserFromMagicCookie] magic cookie is set to be forgotten');
            Cookie::queue(Cookie::forget('_inside_token'));
        } catch (Exception $exception) {
        }
    }

    /**
     * Encrypt or token
     */
    public function encryptToken(string $token): string
    {
        return hash('sha256', $token);
    }

    /**
     * Log the user out
     * @throws Exception
     */
    public function logout(?User $user = null): ?User
    {
        if ($user === null) {
            $user = Auth::user();
        }

        if ($user instanceof User) {
            $bridge = new BridgeAuthentication;
            $bridge->logoutUser($user);

            if ($user->currentAccessToken()) {
                $user->currentAccessToken()->delete();
            }

            if (Cookie::has('_inside_token')) {
                $this->invalidateUserFromMagicCookie();
            }

            Event::dispatch(new AuthenticationLogoutEvent($user));

            StoreIntendedPathService::forget();

            Log::info('Logout successfull', ['uuid' => $user->uuid, 'email' => $user->email]);

            return $user;
        }

        // Cleanup cookie if it exists
        $this->invalidateUserFromMagicCookie();

        Log::error('Logout unsuccessfull, check the api-token header');

        return null;
    }

    /**
     * Force disconnect a $user for $reason
     *
     * @param User $user
     * @param string|null $reason
     * @throws Exception
     */
    public function kickUser(User $user, ?string $reason = null): void
    {
        UserHasBeenDisconnectedEvent::dispatch($user, $reason);

        $bridge = new BridgeAuthentication();
        $bridge->logoutUser($user);

        $user->currentAccessToken()?->delete();

        Log::info('User kicked', ['uuid' => $user->uuid, 'email' => $user->email]);
    }

    /**
     * Force disconnects all active user for $reason
     *
     * @param string|null $reason
     * @throws Exception
     */
    public function kickAll(?string $reason = null): void
    {
        foreach (Token::active()->groupBy(['user_uuid'])->get() as $token) {
            $this->kickUser($token->user, $reason);
        }
    }

    /**
     * check if we have magic cookie
     *
     * @param Request $request
     * @return bool
     */
    public function hasMagicCookie(Request $request): bool
    {
        return Cookie::has('_inside_token');
    }

    /**
     * check that used jwt token is an external valid token ( if $application is set, it check against current
     * application )
     *
     * @param string|null $application
     * @return bool
     */
    public function checkExternalJWTToken(?string $application = null): bool
    {
        return ! auth('jwt')->guest() && auth('jwt')->getPayload()->get('type') === 'external'
            && ($application === null
                || $application === auth('jwt')->user()->name);
    }
}
