<?php

namespace Inside\Authentication\Providers;

use Drupal;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use Inside\Authentication\Console\GenerateApplicationClientAddCommand;
use Inside\Authentication\Console\GenerateApplicationClientListCommand;
use Inside\Authentication\Console\GenerateApplicationClientRemoveCommand;
use Inside\Authentication\Console\GenerateApplicationClientResetCommand;
use Inside\Authentication\Console\GenerateApplicationTokenCommand;
use Inside\Authentication\Console\UsersLogoutCommand;
use Inside\Authentication\Contracts\Authentication as AuthenticationContract;
use Inside\Authentication\Contracts\InsideSessionLifetime;
use Inside\Authentication\Facades\Authentication;
use Inside\Authentication\Http\Middleware\AuthenticationMiddleware;
use Inside\Authentication\Http\Middleware\CaptchaMiddleware;
use Inside\Authentication\Http\Middleware\ExternalJwtClientAuthenticate;
use Inside\Authentication\Http\Middleware\KeepCookieAliveIfNeeded;
use Inside\Authentication\Http\Middleware\MaeciaAdminMiddleware;
use Inside\Authentication\Listeners\AuthenticationEventSubscriber;
use Inside\Authentication\Listeners\UserDisabledListener;
use Inside\Authentication\Models\Token;
use Inside\Authentication\Models\User;
use Inside\Authentication\Services\AuthenticationService;
use Inside\Authentication\Services\InsideSessionLifetimeService;
use Inside\Authentication\Services\SfrSmsSenderService;
use Inside\Authentication\Services\TwoFactorAuthenticationService;
use Inside\Host\Bridge\BridgeAuthentication;
use Inside\Permission\Http\Middlewares\BackofficeEntryMiddleware;
use Inside\Support\EventServiceProvider;
use Inside\User\Events\UserDisabledEvent;
use Psr\Container\ContainerExceptionInterface;
use Psr\Container\NotFoundExceptionInterface;
use Tymon\JWTAuth\Providers\LumenServiceProvider;

/**
 * Inside Authentication service.
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class AuthenticationServiceProvider extends EventServiceProvider
{
    protected array $listen = [
        UserDisabledEvent::class => [
            UserDisabledListener::class,
        ],
    ];

    protected array $subscribe = [
        AuthenticationEventSubscriber::class,
    ];

    /** @var string[] */
    public array $bindings = [
        AuthenticationContract::class => AuthenticationService::class,
        'inside.2fa.authentication' => TwoFactorAuthenticationService::class,
        'inside.sfr.sms.sender' => SfrSmsSenderService::class,
        InsideSessionLifetime::class => InsideSessionLifetimeService::class,
    ];

    /** @var string[] */
    protected array $commands = [
        GenerateApplicationTokenCommand::class,
        GenerateApplicationClientAddCommand::class,
        GenerateApplicationClientListCommand::class,
        GenerateApplicationClientRemoveCommand::class,
        GenerateApplicationClientResetCommand::class,
        UsersLogoutCommand::class,
    ];

    /**
     * Bootstrap the application services.
     *
     * @return void
     */
    public function boot(): void
    {
        parent::boot();

        Auth::viaRequest(
            'api',
            function ($request) {
                // Special drupal case !
                if (array_key_exists('fromDrupal', $GLOBALS) && $GLOBALS['fromDrupal'] === true) {
                    $bridge = new BridgeAuthentication();
                    $uuid = $bridge->current();
                    if (is_null($uuid) && ! is_null($user = Authentication::getUserFromMagicCookie($request))) {
                        $bridge->loginUser($user);
                        $uuid = $bridge->current();
                    }
                    if (! is_null($uuid)) {
                        // user is currently connected to back office
                        $hasAdminAccess = Drupal::currentUser()->hasPermission('administer site configuration');
                        $user = User::where('uuid_host', $uuid)->first();
                        if ($user && $hasAdminAccess) {
                            return $user;
                        }
                    }
                }

                $apiToken = null;
                if (app()->runningInConsole()) {
                    $apiToken = config('auth.magic_cli_token');
                } elseif (app()->routeIsFallback($request)) {
                    return null;
                }
                if (! $apiToken) {
                    $apiToken = $request->header('api-token');
                    if (! $apiToken) {
                        // Check $request has not been update
                        $apiToken = request()->header('api-token');
                    }
                }
                if ($apiToken) {
                    // Find token
                    $accessToken = Token::where('token', Authentication::encryptToken($apiToken))->first();
                    if (! $accessToken || $accessToken->hasExpired()) {
                        // token is invalid or expired make sure cookie is invalidated
                        Authentication::invalidateUserFromMagicCookie();

                        return null;
                    }

                    return $accessToken->user->withAccessToken(
                        tap($accessToken->forceFill(['last_used_at' => now()]))->save()
                    );
                } else {
                    $specialMaeciaAdminAccess = false;
                    if (
                        (config('app.debug', false)
                            && in_array(
                                app()->environment(),
                                ['local', 'preprod', 'dev', 'development']
                            ))
                        || $this->isDirectAccessException($request)
                    ) {
                        // Only available with debug on & environment local or develop!
                        $bridge = new BridgeAuthentication();

                        $uuid = $bridge->current();
                        if ($uuid) {
                            $user = User::where('uuid_host', $uuid)->firstOrFail();

                            // Only for maecia super admin users
                            if (
                                $user->hasRole('super_administrator')
                                && Str::endsWith(
                                    $user->email,
                                    '@maecia.com'
                                )
                            ) {
                                $specialMaeciaAdminAccess = true;
                            }
                        }
                    }

                    if (
                        $this->isDirectAccessException($request)
                        || (in_array(request()->segment(1), config('authentication.magic_cookie_auth_paths', []))
                            || $specialMaeciaAdminAccess)
                    ) {
                        // Special magic for resources protection
                        if ($user = Authentication::getUserFromMagicCookie($request)) {
                            return $user;
                        }
                    }
                }

                return null;
            }
        );

        // Make cli auth be first maecia super admin user
        if (app()->runningInConsole()) {
            /** @var ?User $user */
            $user = User::where('email', 'like', '%@maecia.com')
                ->where('status', true)
                ->whereHas('roles', fn ($query) => $query->where('name', 'super_administrator'))
                ->first();
            if (! is_null($user) && $user->status === true) {
                /** @var ?Token $accessToken */
                $accessToken = $user->tokens()
                    ->where('authenticator', 'cli')
                    ->orderByDesc('created_at')
                    ->first();

                if (is_null($accessToken) || $accessToken->hasExpired()) {
                    $accessToken = $user->createToken('cli');
                } else {
                    $accessToken->update([
                        'token' => Authentication::encryptToken($plainTextToken = Str::random(80)),
                    ]);
                    $accessToken->plainTextToken = $plainTextToken;
                }
                config(['auth.magic_cli_token' => $accessToken->plainTextToken]);
            }
        }

        if (! app()->runningInConsole()) {
            config([
                'session.lifetime' => app(InsideSessionLifetime::class)->getSessionSSOLifetime(),
                'session.longer_lifetime' => app(InsideSessionLifetime::class)->getSessionLongerLifetime(),
                'session.sso_lifetime' => app(InsideSessionLifetime::class)->getSessionSSOLifetime(),
            ]);
        }
    }

    /**
     * Some api call can be called directly & use cookie system authentication instead of the bearer header token
     * A regex or a full path is to be present in direct_access_exceptions authentication config
     */
    protected function isDirectAccessException(Request $request): bool
    {
        $path = $request->getPathInfo();
        if (strlen($path) > 0 && $path[0] == '/') {
            $path = substr($path, 1);
        }
        foreach (config('authentication.direct_access_exceptions', []) as $exception) {
            if (is_string($exception) && strlen($exception) > 0) {
                if ($exception[0] == '#') {
                    // this is a regex
                    if (preg_match($exception, $path)) {
                        return true;
                    }
                } elseif ($exception == $path) {
                    return true;
                }
            }
        }

        return false;
    }

    /**
     * Register the application services.
     */
    public function register(): void
    {
        try {
            $this->mergeConfigFrom(__DIR__.'/../../config/layout_pathes.php', 'layout_pathes');
            $this->mergeConfigFrom(__DIR__.'/../../config/jwt.php', 'jwt');
            $this->mergeRecursiveConfigFrom(__DIR__.'/../../config/authentication.php', 'authentication');
            $this->mergeRecursiveConfigFrom(__DIR__.'/../../config/scheduler.php', 'scheduler');
        } catch (NotFoundExceptionInterface | ContainerExceptionInterface) {
        }

        $this->app->routeMiddleware([
            'captcha' => CaptchaMiddleware::class,
        ]);

        $this->app->register(LumenServiceProvider::class);

        $this->app->middleware([
            KeepCookieAliveIfNeeded::class,
        ]);

        $this->app->routeMiddleware(
            [
                'auth' => \Illuminate\Auth\Middleware\Authenticate::class,
                'auth.api' => AuthenticationMiddleware::class,
                'auth.maecia' => MaeciaAdminMiddleware::class,
                'auth.backoffice.access' => BackofficeEntryMiddleware::class,
                'auth.external' => ExternalJwtClientAuthenticate::class,
            ]
        );
    }
}
