<?php

namespace Inside\Authentication\Models;

use Illuminate\Auth\Authenticatable;
use Illuminate\Auth\Passwords\CanResetPassword;
use Illuminate\Contracts\Auth\Access\Authorizable as AuthorizableContract;
use Illuminate\Contracts\Auth\Authenticatable as AuthenticatableContract;
use Illuminate\Contracts\Auth\CanResetPassword as CanResetPasswordContract;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use Inside\Authentication\Events\UserDecisionized;
use Inside\Authentication\Events\UserUndecisionized;
use Inside\Authentication\Models\Traits\Decisionable;
use Inside\Authentication\Models\Traits\HasTokens;
use Inside\Content\Chat\Models\Chat;
use Inside\Content\Facades\Schema;
use Inside\Content\Models\Content;
use Inside\Content\Models\Contents\Users;
use Inside\Content\Models\Traits\CanRunWithoutEvents;
use Inside\Database\Eloquent\WithEnhancedBuilder;
use Inside\Facades\Inside;
use Inside\Notify\Concerns\Notifiable;
use Inside\Permission\Concerns\Permissible;
use Inside\Permission\Exodus\Dto\Privileges\BackofficePrivilegeDto;
use Inside\Permission\Exodus\Dto\Privileges\ContentPrivilegeDto;
use Inside\Permission\Exodus\Dto\Privileges\ContentTypePrivilegeDto;
use Inside\Permission\Exodus\Dto\Privileges\MenuPrivilegeDto;
use Inside\Permission\Exodus\Enums\CapabilityEnum;
use Inside\Permission\Exodus\Services\RolePrivilegesService;
use Inside\Permission\Models\Traits\HasRoles;
use Inside\Reaction\Models\Reaction;
use Inside\User\Notifications\CreatePasswordNotification;
use Inside\User\Notifications\ResetPasswordNotification;
use Laravel\Lumen\Auth\Authorizable;

/**
 * Authentication user model.
 *
 * @category Class
 * @author Maecia <technique@maecia.com>
 * @license http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link http://www.maecia.com/
 *
 * @property string $uuid
 * @property string $uuid_host
 * @property string $id
 * @property string|null $name
 * @property string|null $email
 * @property string $password
 * @property bool $status
 * @property string|null $langcode
 * @property string|null $remember_token
 * @property int|null $created_at
 * @property int|null $updated_at
 * @property \Illuminate\Support\Carbon|null $decisionable_at
 * @property \Illuminate\Support\Carbon|null $last_login_at
 * @property \Illuminate\Support\Carbon|null $last_access_at
 * @property string|null $provider_type
 * @property string|null $provider_name
 * @property-read string $full_name
 * @property-read Users|null $information
 * @property-read Token $lastActiveToken
 * @property-read \Illuminate\Database\Eloquent\Collection|Chat[] $messages
 * @property-read int|null $messages_count
 * @property-read \Illuminate\Notifications\DatabaseNotificationCollection|\Inside\Notify\Models\Notification[] $notifications
 * @property-read int|null $notifications_count
 * @property-read \Inside\Permission\Models\User $permission
 * @property-read \Illuminate\Database\Eloquent\Collection|Reaction[] $reactions
 * @property-read int|null $reactions_count
 * @property-read \Illuminate\Database\Eloquent\Collection|\Inside\Permission\Models\Role[] $roles
 * @property-read int|null $roles_count
 * @property-read \Illuminate\Database\Eloquent\Collection|\Inside\Notify\Models\NotificationType[] $subscriptions
 * @property-read int|null $subscriptions_count
 * @property-read \Illuminate\Database\Eloquent\Collection|\Inside\Authentication\Models\Token[] $tokens
 * @property-read int|null $tokens_count
 * @property-read ?string $firstname
 * @property-read ?string $lastname
 * @method static \Inside\Database\Eloquent\Builder|User newModelQuery()
 * @method static \Inside\Database\Eloquent\Builder|User newQuery()
 * @method static \Inside\Database\Eloquent\Builder|User query()
 * @method static \Inside\Database\Eloquent\Builder|User whereHas(string $relation, \Closure $callback = null, string $operator = '>=', int $count = 1)
 * @method static \Inside\Database\Eloquent\Builder select($columns = ['*'])
 * @method static \Inside\Database\Eloquent\Builder whereNotIn(string $column, mixed $values, string $boolean = 'and')
 * @method static Builder|User whereCreatedAt($value)
 * @method static Builder|User whereDecisionableAt($value)
 * @method static Builder|User whereEmail($value)
 * @method static Builder|User whereLangcode($value)
 * @method static Builder|User whereLastLoginAt($value)
 * @method static Builder|User whereName($value)
 * @method static Builder|User wherePassword($value)
 * @method static Builder|User whereProviderName($value)
 * @method static Builder|User whereProviderType($value)
 * @method static Builder|User whereRememberToken($value)
 * @method static Builder|User whereStatus($value)
 * @method static Builder|User whereUpdatedAt($value)
 * @method static Builder|User whereUuid($value)
 * @method static Builder|User whereUuidHost($value)
 *
 * @mixin \Eloquent
 * @mixin Builder
 */
class User extends Model implements AuthenticatableContract, AuthorizableContract, CanResetPasswordContract
{
    use Authenticatable;
    use Authorizable;
    use Notifiable;
    use CanResetPassword;
    use HasTokens;
    use WithEnhancedBuilder;
    use Decisionable;
    use CanRunWithoutEvents;
    use HasRoles;

    /**
     * The table without the timestamps.
     *
     * @var bool
     */
    public $timestamps = false;

    /**
     * The primary key is not incrementable.
     *
     * @var bool
     */
    public $incrementing = false;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'inside_users';

    /**
     * The table primary key.
     *
     * @var string
     */
    protected $primaryKey = 'uuid';

    /**
     * The type of the primary key
     *
     * @var string
     */
    protected $keyType = 'string';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'name',
        'email',
        'password',
        'uuid',
        'remember_token',
        'last_login_at',
        'last_access_at',
    ];

    /**
     * The attributes excluded from the model's JSON form.
     *
     * @var array
     */
    protected $hidden = [
        'id',
        'password',
        'remember_token',
    ];

    /**
     * The date attributes
     *
     * @var array
     */
    protected $dates = [
        'created_at',
        'updated_at',
        'last_login_at',
        'last_access_at',
    ];

    /**
     * Casts the following attributes
     *
     * @var array
     */
    protected $casts = [
        'uuid' => 'string',
        'status' => 'boolean',
        'created_at' => 'timestamp',
        'updated_at' => 'timestamp',
    ];

    protected static function booted(): void
    {
        static::decisionalized(
            function ($user) {
                UserDecisionized::dispatch($user);
            }
        );
        static::undecisionalized(
            function ($user) {
                UserUndecisionized::dispatch($user);
            }
        );
    }

    /**
     * Mutator to hash password attribute.
     *
     * @param string $password
     * @return void
     */
    public function setPasswordAttribute(string $password): void
    {
        $this->attributes['password'] = Hash::make($password);
    }

    /**
     * Retrieve user information
     *
     * @return HasOne
     */
    public function information(): HasOne
    {
        return $this->hasOne(Users::class, 'uuid', 'uuid');
    }

    /**
     * Send our password notification via email
     *
     * @param string $token
     * @see ResetPasswordNotification
     */
    public function sendPasswordResetNotification($token): void
    {
        if (! is_null($this->email)) {
            $this->notify(new ResetPasswordNotification($token, $this->email));
        }
    }

    /**
     * Send our password notification via email
     *
     * @param string $token
     * @see ResetPasswordNotification
     */
    public function sendPasswordCreateNotification(string $token): void
    {
        if (! is_null($this->email)) {
            $this->notify(new CreatePasswordNotification($token, $this->email));
        }
    }

    /**
     * Chat messages
     *
     * @return HasMany
     */
    public function messages(): HasMany
    {
        return $this->hasMany(Chat::class);
    }

    /**
     * Retrieve user permission model
     *
     * @return HasOne
     * @deprecated use this model instead
     */
    public function permission(): HasOne
    {
        return $this->hasOne(\Inside\Permission\Models\User::class, 'uuid', 'uuid');
    }

    /**
     * Does this user is a Super administrator ( meaning he can do whatever he wants !!! )
     *
     * @return bool
     */
    public function isSuperAdmin(): bool
    {
        return $this->hasAnyRole('super_administrator'); // || php_sapi_name() == 'cli';
    }

    /**
     * Does this user is a maecia super admin ?
     *
     * @return bool
     */
    public function isMaeciaAdmin(): bool
    {
        return $this->hasAnyRole('super_administrator') && $this->email && Str::endsWith($this->email, '@maecia.com');
    }

    /**
     * get all reactions for this user
     *
     * @return HasMany
     */
    public function reactions(): HasMany
    {
        return $this->hasMany(
            Reaction::class,
            'user_uuid'
        );
    }

    /**
     * get nice full_name
     *
     * @return string
     */
    public function getFullNameAttribute(): string
    {
        $fullName = '';
        $title = '';
        if (Schema::hasField('users', 'title')) {
            $title = trim($this->information->title);

            if (! empty($title)) {
                $fullName .= Str::ucfirst(Str::lower($title));
            }
        }
        $firstname = trim($this->information->firstname);
        if (! empty($firstname)) {
            if (! empty($title)) {
                $fullName .= ' ';
            }
            $fullName .= Str::ucfirst(Str::lower($firstname));
        }
        $lastname = trim($this->information->lastname);
        if (! empty($lastname)) {
            if (! empty($firstname) || ! empty($title)) {
                $fullName .= ' ';
            }
            $fullName .= Str::upper($lastname);
        }
        if (empty($fullName)) {
            return ' - ';
        }

        return $fullName;
    }

    /**
     * Magic getter to user information
     *
     * @param string $key
     * @return mixed|void
     */
    public function __get($key)
    {
        if (parent::__isset($key)) {
            return parent::__get($key);
        }

        if (isset($this->information()->first()->{$key})) {
            return $this->information()->first()->{$key};
        }
    }

    /**
     * Determine if a magic attribute exists
     *
     * @param string $key
     * @return bool
     */
    public function __isset($key)
    {
        if (parent::__isset($key)) {
            return true;
        }

        return isset($this->information()->first()->{$key});
    }

    public function getRolePrivilegesService(): RolePrivilegesService
    {
        return RolePrivilegesService::of(...$this->roles);
    }

    /**
     * @return Collection<ContentTypePrivilegeDto>
     */
    public function getAuthorizedContentTypePrivileges(): Collection
    {
        return $this->getRolePrivilegesService()->getContentTypePrivileges()->filter->isAuthorized();
    }

    /**
     * @return Collection<ContentPrivilegeDto>
     */
    public function getAuthorizedCategorizableContentPrivileges(): Collection
    {
        return $this->getRolePrivilegesService()->getCategorizableContentPrivileges()->filter->isAuthorized();
    }

    /**
     * @return Collection<MenuPrivilegeDto>
     */
    public function getAuthorizedMenuPrivileges(): Collection
    {
        return $this->getRolePrivilegesService()->getMenuPrivileges()->filter->isAuthorized();
    }

    /**
     * @return Collection<string>
     */
    public function getAuthorizedBackofficePrivileges(): Collection
    {
        if ($this->isSuperAdmin()) {
            return Inside::getAllBackofficeEntries()->merge(['permission']);
        }

        return $this->getRolePrivilegesService()
            ->getBackofficePrivileges()
            ->filter->isAuthorized()
            ->map(fn (BackofficePrivilegeDto $dto) => strtolower($dto->getName()))
            ->flatten();
    }

    /**
     * @param string $capability
     * @param Content|string $class
     * @return bool
     */
    public function hasContentTypePrivilegeTo(string $capability, Content|string $class): bool
    {
        return $this
            ->getAuthorizedContentTypePrivileges()
            ->map(fn (ContentTypePrivilegeDto $dto) => $dto->toArray())
            ->where('capability.name', $capability)
            ->where('type', $class instanceof Content ? $class::class : $class)
            ->isNotEmpty();
    }

    public function hasBackofficeAccessTo(string $backoffice): bool
    {
        return $this->getAuthorizedBackofficePrivileges()->contains($backoffice);
    }
}
