<?php

declare(strict_types=1);

namespace Inside\Authentication\Models\Traits;

use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Str;
use Inside\Authentication\AccessToken;
use Inside\Authentication\Facades\Authentication;
use Inside\Authentication\Models\Token;
use Inside\Authentication\Models\User;
use Inside\DeviceInformation;

trait HasTokens
{
    /**
     * Current used token
     */
    protected ?Token $accessToken = null;

    public function lastActiveToken(): BelongsTo
    {
        return $this->belongsTo(Token::class, 'uuid', 'user_uuid');
    }

    /**
     * create a token for $authenticator with $abilities
     */
    public function createToken(string $authenticator, array $abilities = ['*'], bool $longerLifetime = false): AccessToken
    {
        $device = new DeviceInformation;

        /** @var Token $token */
        $token = $this->tokens()->create(
            [
                'token'           => Authentication::encryptToken($plainTextToken = Str::random(80)),
                'authenticator'   => $authenticator,
                'abilities'       => $abilities,
                'device'          => $device->toArray(),
                'last_login_ip'   => request()->getClientIp(),
                'longer_lifetime' => $longerLifetime,
            ]
        );

        return new AccessToken($token, $plainTextToken);
    }

    /**
     * Get all tokens
     */
    public function tokens(): HasMany
    {
        return $this->hasMany(Token::class, 'user_uuid', 'uuid');
    }

    /**
     * Use accessToken
     */
    public function withAccessToken(Token $accessToken): self
    {
        $this->accessToken = $accessToken;

        return $this;
    }

    /**
     * Get current accessToken
     */
    public function currentAccessToken(): ?Token
    {
        return $this->accessToken;
    }
}
