<?php

namespace Inside\Authentication\Models\Traits;

use Closure;
use Illuminate\Database\Eloquent\Builder;

/**
 * Trait Decisionable
 */
trait Decisionable
{
    /**
     * Initialization
     */
    public function initializeDecisionable(): void
    {
        if (! isset($this->casts[$this->getDecisionableAtColumn()])) {
            $this->casts[$this->getDecisionableAtColumn()] = 'datetime';
        }
    }

    /**
     * add decisionalizing event callback
     */
    public static function decisionalizing(Closure $callback): void
    {
        static::registerModelEvent('decisionalizing', $callback);
    }

    /**
     * add decisionalized event callback
     */
    public static function decisionalized(Closure $callback): void
    {
        static::registerModelEvent('decisionalized', $callback);
    }

    /**
     * add undecisionalizing event callback
     */
    public static function undecisionalizing(Closure $callback): void
    {
        static::registerModelEvent('undecisionalizing', $callback);
    }

    /**
     * add undecisionalized event callback
     */
    public static function undecisionalized(Closure $callback): void
    {
        static::registerModelEvent('undecisionalized', $callback);
    }

    /**
     * sync driver is not support due to synchronised events
     */
    protected function isDecisionable(): bool
    {
        return (config('users.decommissioning', false) === true) && config('queue.default', 'sync') != 'sync'
            && ! $this->information->is_maintenance
            && $this->permission !== null
            && ! $this->permission->hasAnyRole(config('users.roles_whitelist', []));
    }

    /**
     * On perform update
     */
    protected function performUpdate(Builder $query): bool
    {
        if ($this->isDecisionable() && $this->isDirty($this->getStatusColumn())) {
            $this->syncStatus($query, $this->{$this->getStatusColumn()});
            if (! $this->{$this->getStatusColumn()}) {
                $this->runDecisionalize($query);
            } else {
                $this->runUndecisionalize($query);
            }
        }

        return parent::performUpdate($query);
    }

    /**
     * Sync status
     */
    protected function syncStatus(Builder $query, bool $status): void
    {
        $this->setKeysForSaveQuery($query)->update(
            [$this->getStatusColumn() => $status]
        );
        $this->changes[$this->getStatusColumn()] = $status;
    }

    /**
     * run Undecisionalize
     */
    protected function runUndecisionalize(Builder $query): bool
    {
        if ($this->fireModelEvent('undecisionalizing') === false) {
            return false;
        }

        $this->{$this->getDecisionableAtColumn()} = null;

        $this->setKeysForSaveQuery($query)->update(
            [$this->getDecisionableAtColumn() => null]
        );
        $this->changes[$this->getDecisionableAtColumn()] = null;

        $this->fireModelEvent('undecisionalized', false);

        return true;
    }

    /**
     * runDecisionalize
     */
    protected function runDecisionalize(Builder $query): bool
    {
        if ($this->fireModelEvent('decisionalizing') === false) {
            return false;
        }

        $this->{$this->getDecisionableAtColumn()} = $this->freshTimestamp();

        $this->setKeysForSaveQuery($query)->update(
            [$this->getDecisionableAtColumn() => $this->{$this->getDecisionableAtColumn()}]
        );
        $this->changes[$this->getDecisionableAtColumn()] = $this->{$this->getDecisionableAtColumn()};

        $this->fireModelEvent('decisionalized', false);

        return true;
    }

    public function getDecisionableAtColumn(): string
    {
        return defined('static::DECISIONABLE_AT') ? static::DECISIONABLE_AT : 'decisionable_at';
    }

    public function getQualifiedDecisionableAtColumn(): string
    {
        return $this->qualifyColumn($this->getDecisionableAtColumn());
    }

    public function getStatusColumn(): string
    {
        return defined('static::STATUS') ? static::STATUS : 'status';
    }

    public function getQualifiedStatusColumn(): string
    {
        return $this->qualifyColumn($this->getStatusColumn());
    }

    protected function getDecommissionedAttribute(): bool
    {
        return $this->getAttribute($this->getDecisionableAtColumn()) !== null;
    }
}
