<?php

namespace Inside\Authentication\Http\Middleware;

use Closure;
use Illuminate\Auth\AuthenticationException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use Inside\Kernel\Authentication\StoreIntendedPathService;

/**
 * Inside Authentication middleware.
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class AuthenticationMiddleware
{
    /**
     * The authorized URIs
     *
     * @var array
     */
    private $authorized = [
        'api/v1/authentication/check'              => 'GET',
        'api/v1/authentication/login'              => 'POST',
        'api/v1/authentication/logout'             => 'POST',
        'api/v1/layout?path=login'                 => 'GET',
        'api/v1/layout?path=register'              => 'GET',
        'api/v1/layout?path=password/reset'        => 'GET',
        'api/v1/layout?path=password/create'       => 'GET',
        'api/v1/layout/build?path=login'           => 'GET',
        'api/v1/layout/build?path=register'        => 'GET',
        'api/v1/layout/build?path=password/reset'  => 'GET',
        'api/v1/layout/build?path=password/create' => 'GET',
        'api/v1/users/register'                    => 'POST',
        'api/v1/password/reset'                    => 'PUT',
        'api/v1/password/email'                    => 'POST',
        'api/v1/password/create/email'             => 'POST',
        'api/v1/token/validate'                    => 'POST',
        'api/v1/public/'                           => 'GET',
        'api/v1/yes'                               => ['GET', 'POST', 'PUT', 'DELETE'],
        'api/v1/settings'                          => 'GET',
        'api/v1/options/colors'                    => 'GET',

        'windows/login'     => 'GET',
        'google/login'      => 'GET',
        'saml/login'        => 'GET',
        'saml/logout'       => 'GET',
        'saml/acs'          => 'POST',
        'saml/sls'          => 'GET',
        'saml/metadata'     => 'GET',
        'terminal'          => 'GET',
        'terminal/endpoint' => 'POST',
    ];

    /**
     * AuthenticationMiddleware constructor.
     */
    public function __construct()
    {
        // Add config specifics authorized routes
        $this->authorized = array_merge(config('authorized', []), $this->authorized);
    }

    /**
     * Handle an incoming request.
     *
     * @param  Request  $request
     * @param  Closure  $next
     *
     * @return mixed
     * @throws AuthenticationException
     */
    public function handle(Request $request, Closure $next): mixed
    {
        if (! Auth::check() && ! $this->isAuthorized($request)) {
            if (request()->is('api/v1/content/*/infos')) {
                $slug = request()->segment(4);
                StoreIntendedPathService::store($slug);
            }

            throw new AuthenticationException();
        }

        return $next($request);
    }

    /**
     * Check if a given URI can bypass authentication validation
     *
     * @param  Request  $request
     * @return bool
     */
    private function isAuthorized(Request $request): bool
    {
        $host = filter_input(INPUT_SERVER, 'HTTP_HOST', FILTER_SANITIZE_URL);
        $https = filter_input(INPUT_SERVER, 'HTTPS', FILTER_SANITIZE_URL);

        $base = $https === 'on' ? 'https' : 'http';
        $host = $base.'://'.$host;
        // Save query string
        if (null !== $qs = $request->getQueryString()) {
            $qs = '?'.$qs;
        }
        // Clean & prepare request Uri
        /** @var string $path */
        $path = preg_replace('/(\/+)/', '/', $request->getPathInfo());
        $requestUri =
            trim(urldecode($request->getUriForPath($path).$qs), '/');

        if ($host == $requestUri) {
            return true;
        }

        if (! Str::contains($requestUri, $host.'/api/v')) {
            return true;
        }

        foreach ($this->authorized as $uri => $methods) {
            if (! is_array($methods)) {
                $methods = [$methods];
            }
            if (Str::contains($requestUri, $uri) && in_array($request->getMethod(), $methods)) {
                return true;
            }
        }

        return false;
    }
}
