<?php

namespace Inside\Authentication\Http\Controllers;

use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Inside\Authentication\Facades\Authentication;
use Laravel\Lumen\Routing\Controller;
use OpenApi\Annotations as OA;
use Tymon\JWTAuth\Exceptions\JWTException;

/**
 * Inside External Authentication controller.
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class ExternalAuthenticationController extends Controller
{
    /**
     * @OA\Post(
     *      path="/external/authentication/login",
     *      operationId="externalAuthenticationLogin",
     *      tags={"Authentification"},
     *      summary="Système d'authentification externe pour obtenir un token",
     *     @OA\Parameter(
     *         name="name",
     *         in="query",
     *         description="Nom du client ( applicatif externe ) souhaitant obtenir un token",
     *         required=true,
     *         example="cypress",
     *         @OA\Schema(
     *             type="string"
     *         )
     *     ),
     *     @OA\Parameter(
     *         name="secret",
     *         in="query",
     *         description="La clé secrete d'identification",
     *         required=true,
     *         example="7hGOCIiqI6qiEcu8Rcd1Elz6lvxCn8wcVxxg6pobfTEqdVEVEEXtx1vVySRbMsmW",
     *         @OA\Schema(
     *             type="string"
     *         )
     *     ),
     *      @OA\Response(
     *          response=200,
     *          description="Le token a utiliser pour consommer l'api",
     *          @OA\JsonContent(ref="#/components/schemas/JWTTokenInformation")
     *     ),
     *      @OA\Response(
     *          response=401,
     *          description="Credential incorrecte"
     *     ),
     *      @OA\Response(
     *          response=500,
     *          description="Impossible de créer le token"
     *     )
     *  )
     */
    public function login(Request $request): JsonResponse
    {
        $credentials = [
            'name'     => $request->get('name'),
            'password' => $request->get('secret'),
        ];
        try {
            if (! $token = auth('jwt')->claims(['type' => 'external'])->attempt($credentials)) {
                return response()->json(['error' => 'invalid_credentials'], 401);
            }
        } catch (JWTException $e) {
            return response()->json(['error' => 'could_not_create_token'], 500);
        }

        return $this->respondWithToken($token);
    }

    /**
     * @OA\Schema(
     *            schema="JWTTokenInformation",
     *            type="object",
     *     @OA\Property(
     *     property="access_token",
     *     description="le token d'accès générer. Pour consommer l'api, il faut désormais utiliser ce token dans
     *     l'entête Authorization: Bearer eyJ0eXAiOiJKV1QiLCJhbGciOiJIUzI1NiJ9.eyJpc3M....",
     *     example="eyJ0eXAiOiJKV1QiLCJhbGciOiJIUzI1NiJ9.eyJpc3MiOiJodHRwOlwvXC9zaWQyLnRlc3RcL2FwaVwvdjFcL2V4dGVybmFsXC9hdXRoZW50aWNhdGlvblwvbG9naW4iLCJpYXQiOjE2MjQ4NzQzMzUsImV4cCI6MTYyNDg3NzkzNSwibmJmIjoxNjI0ODc0MzM1LCJqdGkiOiJKaFBwWjhrOVdldkxMcldEIiwic3ViIjozLCJwcnYiOiI2ZWNkM2Q3YWUzODVmYzlhYjI0ZGJkZjk1ZjJiMzVlOWMyYmFjNDA1In0.14hhCXe5BZXe8VW__36VPki3Czi0HE3GBGIh-jGFkaI",
     *     type="string"
     *     ),
     *     @OA\Property(
     *     property="token_type",
     *     description="Type de token ( pour le moment, seul le token bearer est supporté )",
     *     type="string"
     *     ),
     *     @OA\Property(
     *     property="expires_in",
     *     description="Durée de vie du token en secondes. Une fois ce délais passé, le token n'est plus valide et ne
     *     permet plus de consommer l'api.", type="integer", example=3600
     *     )
     *  )
     *
     * @param string $token
     * @return JsonResponse
     */
    protected function respondWithToken(string $token): JsonResponse
    {
        return response()->json(
            [
                'access_token' => $token,
                'token_type'   => 'bearer',
                'expires_in'   => auth('jwt')->factory()->getTTL() * 60,
            ]
        );
    }

    /**
     * @OA\Post(
     *      path="/external/authentication/logout",
     *      operationId="externalAuthenticationLogout",
     *      tags={"Authentification"},
     *      summary="Système d'invalidation du token courant. Le token sera blacklisté et ne pourra plus être utilisé,
     *      même pour être rafraichi.",
     *      @OA\Response(
     *          response=200,
     *          description="Le nouveau token a utiliser pour consommer l'api"
     *     ),
     *      @OA\Response(
     *          response=401,
     *          description="Le token utilisé est incorrecte"
     *     ),
     *      @OA\Response(
     *          response=500,
     *          description="Impossible de blacklister le token"
     *     )
     *  )
     *
     *
     * @return JsonResponse
     */
    public function logout(): JsonResponse
    {
        auth('jwt')->logout();

        return response()->json(['message' => 'bye']);
    }

    /**
     * @OA\Post(
     *      path="/external/authentication/refresh",
     *      operationId="externalAuthenticationRefresh",
     *      tags={"Authentification"},
     *      summary="Rafraichissement du token d'accès. Note: le token actuel sera blacklisté, une fois le token
     *      rafraichi il faut nécessairement utiliser le nouveau token pour consommer l'api.",
     *      @OA\Response(
     *          response=200,
     *          description="Le nouveau token a utiliser pour la connection",
     *          @OA\JsonContent(ref="#/components/schemas/JWTTokenInformation")
     *     ),
     *      @OA\Response(
     *          response=401,
     *          description="Le token utilisé est incorrecte"
     *     ),
     *      @OA\Response(
     *          response=500,
     *          description="Impossible de créer un nouveau token"
     *     )
     *  )
     *
     * @return JsonResponse
     */
    public function refresh(): JsonResponse
    {
        return $this->respondWithToken(auth('jwt')->refresh());
    }

    /**
     * @OA\Post(
     *      path="/external/authentication/check",
     *      operationId="externalAuthenticationCheck",
     *      tags={"Authentification"},
     *      summary="Vérification du token d'accès.",
     *      @OA\Response(
     *          response=200,
     *          description="Indique si le token utilisé est valide ou non. Note: seul les token générer par la console
     *          inside:authentication:client:add sont considérés valide. Les tokens systèmes ne le sont pas."
     *     ),
     *      @OA\Response(
     *          response=401,
     *          description="Le token utilisé est incorrecte"
     *     )
     *  )
     *
     * @return JsonResponse
     */
    public function check(): JsonResponse
    {
        return response()->json(
            ['status' => Authentication::checkExternalJWTToken()]
        );
    }
}
